package org.wildfly.swarm.config.jgroups;

import org.wildfly.swarm.config.runtime.Address;
import org.wildfly.swarm.config.runtime.ResourceType;
import java.beans.PropertyChangeSupport;
import java.beans.PropertyChangeListener;
import org.wildfly.swarm.config.runtime.ModelNodeBinding;
import java.util.List;
import org.wildfly.swarm.config.runtime.Subresource;
import org.wildfly.swarm.config.jgroups.ProtocolConsumer;
import org.wildfly.swarm.config.jgroups.ProtocolSupplier;
import org.wildfly.swarm.config.jgroups.Protocol;
import org.wildfly.swarm.config.jgroups.stack.TransportConsumer;
import org.wildfly.swarm.config.jgroups.stack.TransportSupplier;
import org.wildfly.swarm.config.jgroups.stack.Transport;
import org.wildfly.swarm.config.jgroups.stack.relay.RELAY2;
import org.wildfly.swarm.config.jgroups.stack.relay.RELAY2Consumer;
import org.wildfly.swarm.config.jgroups.stack.relay.RELAY2Supplier;
/**
 * The configuration of a JGroups protocol stack.
 */
@Address("/subsystem=jgroups/stack=*")
@ResourceType("stack")
public class Stack<T extends Stack<T>> {

	private String key;
	private PropertyChangeSupport pcs;
	private StackResources subresources = new StackResources();

	public Stack(String key) {
		this.key = key;
	}

	public String getKey() {
		return this.key;
	}

	/**
	 * Adds a property change listener
	 */
	public void addPropertyChangeListener(PropertyChangeListener listener) {
		if (null == this.pcs)
			this.pcs = new PropertyChangeSupport(this);
		this.pcs.addPropertyChangeListener(listener);
	}

	/**
	 * Removes a property change listener
	 */
	public void removePropertyChangeListener(PropertyChangeListener listener) {
		if (this.pcs != null)
			this.pcs.removePropertyChangeListener(listener);
	}

	public StackResources subresources() {
		return this.subresources;
	}

	/**
	 * Add all Protocol objects to this subresource
	 * 
	 * @return this
	 * @param value
	 *            List of Protocol objects.
	 */
	@SuppressWarnings("unchecked")
	public T protocols(List<Protocol> value) {
		this.subresources.protocols = value;
		return (T) this;
	}

	/**
	 * Add the Protocol object to the list of subresources
	 * 
	 * @param value
	 *            The Protocol to add
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T protocol(Protocol value) {
		this.subresources.protocols.add(value);
		return (T) this;
	}

	/**
	 * Create and configure a Protocol object to the list of subresources
	 * 
	 * @param key
	 *            The key for the Protocol resource
	 * @param config
	 *            The ProtocolConsumer to use
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T protocol(String childKey, ProtocolConsumer consumer) {
		Protocol<?> child = new Protocol<>(childKey);
		if (consumer != null) {
			consumer.accept(child);
		}
		protocol(child);
		return (T) this;
	}

	/**
	 * Create and configure a Protocol object to the list of subresources
	 * 
	 * @param key
	 *            The key for the Protocol resource
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T protocol(String childKey) {
		protocol(childKey, null);
		return (T) this;
	}

	/**
	 * Install a supplied Protocol object to the list of subresources
	 */
	@SuppressWarnings("unchecked")
	public T protocol(ProtocolSupplier supplier) {
		protocol(supplier.get());
		return (T) this;
	}

	/**
	 * Add all Transport objects to this subresource
	 * 
	 * @return this
	 * @param value
	 *            List of Transport objects.
	 */
	@SuppressWarnings("unchecked")
	public T transports(List<Transport> value) {
		this.subresources.transports = value;
		return (T) this;
	}

	/**
	 * Add the Transport object to the list of subresources
	 * 
	 * @param value
	 *            The Transport to add
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T transport(Transport value) {
		this.subresources.transports.add(value);
		return (T) this;
	}

	/**
	 * Create and configure a Transport object to the list of subresources
	 * 
	 * @param key
	 *            The key for the Transport resource
	 * @param config
	 *            The TransportConsumer to use
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T transport(String childKey, TransportConsumer consumer) {
		Transport<?> child = new Transport<>(childKey);
		if (consumer != null) {
			consumer.accept(child);
		}
		transport(child);
		return (T) this;
	}

	/**
	 * Create and configure a Transport object to the list of subresources
	 * 
	 * @param key
	 *            The key for the Transport resource
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T transport(String childKey) {
		transport(childKey, null);
		return (T) this;
	}

	/**
	 * Install a supplied Transport object to the list of subresources
	 */
	@SuppressWarnings("unchecked")
	public T transport(TransportSupplier supplier) {
		transport(supplier.get());
		return (T) this;
	}

	/**
	 * The configuration of a RELAY protocol
	 */
	@SuppressWarnings("unchecked")
	public T rELAY2(RELAY2 value) {
		this.subresources.rELAY2 = value;
		return (T) this;
	}

	/**
	 * The configuration of a RELAY protocol
	 */
	@SuppressWarnings("unchecked")
	public T rELAY2(RELAY2Consumer consumer) {
		RELAY2<?> child = new RELAY2<>();
		if (consumer != null) {
			consumer.accept(child);
		}
		this.subresources.rELAY2 = child;
		return (T) this;
	}

	/**
	 * The configuration of a RELAY protocol
	 */
	@SuppressWarnings("unchecked")
	public T rELAY2() {
		RELAY2<?> child = new RELAY2<>();
		this.subresources.rELAY2 = child;
		return (T) this;
	}

	/**
	 * The configuration of a RELAY protocol
	 */
	@SuppressWarnings("unchecked")
	public T rELAY2(RELAY2Supplier supplier) {
		this.subresources.rELAY2 = supplier.get();
		return (T) this;
	}

	/**
	 * Child mutators for Stack
	 */
	public static class StackResources {
		/**
		 * The configuration of a protocol within a protocol stack.
		 */
		private List<Protocol> protocols = new java.util.ArrayList<>();
		/**
		 * The configuration of a transport for a protocol stack.
		 */
		private List<Transport> transports = new java.util.ArrayList<>();
		private RELAY2 rELAY2;

		/**
		 * Get the list of Protocol resources
		 * 
		 * @return the list of resources
		 */
		@Subresource
		public List<Protocol> protocols() {
			return this.protocols;
		}

		/**
		 * Get the list of Transport resources
		 * 
		 * @return the list of resources
		 */
		@Subresource
		public List<Transport> transports() {
			return this.transports;
		}

		/**
		 * The configuration of a RELAY protocol
		 */
		@Subresource
		public RELAY2 rELAY2() {
			return this.rELAY2;
		}
	}
}