package org.wildfly.swarm.config.remoting;

import org.wildfly.swarm.config.runtime.Address;
import org.wildfly.swarm.config.runtime.ResourceType;
import java.beans.PropertyChangeSupport;
import java.beans.PropertyChangeListener;
import org.wildfly.swarm.config.runtime.ModelNodeBinding;
import java.util.List;
import org.wildfly.swarm.config.runtime.Subresource;
import org.wildfly.swarm.config.remoting.PropertyConsumer;
import org.wildfly.swarm.config.remoting.PropertySupplier;
import org.wildfly.swarm.config.remoting.Property;
/**
 * Remoting outbound connection with an implicit remote:// URI scheme.
 */
@Address("/subsystem=remoting/remote-outbound-connection=*")
@ResourceType("remote-outbound-connection")
public class RemoteOutboundConnection<T extends RemoteOutboundConnection<T>> {

	private String key;
	private PropertyChangeSupport pcs;
	private String outboundSocketBindingRef;
	private String protocol;
	private String securityRealm;
	private String username;
	private RemoteOutboundConnectionResources subresources = new RemoteOutboundConnectionResources();

	public RemoteOutboundConnection(String key) {
		this.key = key;
	}

	public String getKey() {
		return this.key;
	}

	/**
	 * Adds a property change listener
	 */
	public void addPropertyChangeListener(PropertyChangeListener listener) {
		if (null == this.pcs)
			this.pcs = new PropertyChangeSupport(this);
		this.pcs.addPropertyChangeListener(listener);
	}

	/**
	 * Removes a property change listener
	 */
	public void removePropertyChangeListener(PropertyChangeListener listener) {
		if (this.pcs != null)
			this.pcs.removePropertyChangeListener(listener);
	}

	/**
	 * Name of the outbound-socket-binding which will be used to determine the
	 * destination address and port for the connection.
	 */
	@ModelNodeBinding(detypedName = "outbound-socket-binding-ref")
	public String outboundSocketBindingRef() {
		return this.outboundSocketBindingRef;
	}

	/**
	 * Name of the outbound-socket-binding which will be used to determine the
	 * destination address and port for the connection.
	 */
	@SuppressWarnings("unchecked")
	public T outboundSocketBindingRef(String value) {
		Object oldValue = this.outboundSocketBindingRef;
		this.outboundSocketBindingRef = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("outboundSocketBindingRef", oldValue,
					value);
		return (T) this;
	}

	/**
	 * The protocol to use for the remote connection. Defaults to http-remoting.
	 */
	@ModelNodeBinding(detypedName = "protocol")
	public String protocol() {
		return this.protocol;
	}

	/**
	 * The protocol to use for the remote connection. Defaults to http-remoting.
	 */
	@SuppressWarnings("unchecked")
	public T protocol(String value) {
		Object oldValue = this.protocol;
		this.protocol = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("protocol", oldValue, value);
		return (T) this;
	}

	/**
	 * Reference to the security realm to use to obtain the password and SSL
	 * configuration.
	 */
	@ModelNodeBinding(detypedName = "security-realm")
	public String securityRealm() {
		return this.securityRealm;
	}

	/**
	 * Reference to the security realm to use to obtain the password and SSL
	 * configuration.
	 */
	@SuppressWarnings("unchecked")
	public T securityRealm(String value) {
		Object oldValue = this.securityRealm;
		this.securityRealm = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("securityRealm", oldValue, value);
		return (T) this;
	}

	/**
	 * The user name to use when authenticating against the remote server.
	 */
	@ModelNodeBinding(detypedName = "username")
	public String username() {
		return this.username;
	}

	/**
	 * The user name to use when authenticating against the remote server.
	 */
	@SuppressWarnings("unchecked")
	public T username(String value) {
		Object oldValue = this.username;
		this.username = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("username", oldValue, value);
		return (T) this;
	}

	public RemoteOutboundConnectionResources subresources() {
		return this.subresources;
	}

	/**
	 * Add all Property objects to this subresource
	 * 
	 * @return this
	 * @param value
	 *            List of Property objects.
	 */
	@SuppressWarnings("unchecked")
	public T properties(List<Property> value) {
		this.subresources.properties = value;
		return (T) this;
	}

	/**
	 * Add the Property object to the list of subresources
	 * 
	 * @param value
	 *            The Property to add
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T property(Property value) {
		this.subresources.properties.add(value);
		return (T) this;
	}

	/**
	 * Create and configure a Property object to the list of subresources
	 * 
	 * @param key
	 *            The key for the Property resource
	 * @param config
	 *            The PropertyConsumer to use
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T property(String childKey, PropertyConsumer consumer) {
		Property<?> child = new Property<>(childKey);
		if (consumer != null) {
			consumer.accept(child);
		}
		property(child);
		return (T) this;
	}

	/**
	 * Create and configure a Property object to the list of subresources
	 * 
	 * @param key
	 *            The key for the Property resource
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T property(String childKey) {
		property(childKey, null);
		return (T) this;
	}

	/**
	 * Install a supplied Property object to the list of subresources
	 */
	@SuppressWarnings("unchecked")
	public T property(PropertySupplier supplier) {
		property(supplier.get());
		return (T) this;
	}

	/**
	 * Child mutators for RemoteOutboundConnection
	 */
	public static class RemoteOutboundConnectionResources {
		/**
		 * Properties supported by the underlying provider. The property name is
		 * inferred from the last element of the properties address.
		 */
		private List<Property> properties = new java.util.ArrayList<>();

		/**
		 * Get the list of Property resources
		 * 
		 * @return the list of resources
		 */
		@Subresource
		public List<Property> properties() {
			return this.properties;
		}
	}
}