package org.wildfly.swarm.config.undertow.configuration.mod_cluster;

import org.wildfly.swarm.config.runtime.Address;
import org.wildfly.swarm.config.runtime.ResourceType;
import java.beans.PropertyChangeSupport;
import java.beans.PropertyChangeListener;
import org.wildfly.swarm.config.runtime.ModelNodeBinding;
import java.util.List;
import org.wildfly.swarm.config.runtime.Subresource;
import org.wildfly.swarm.config.undertow.configuration.mod_cluster.balancer.NodeConsumer;
import org.wildfly.swarm.config.undertow.configuration.mod_cluster.balancer.NodeSupplier;
import org.wildfly.swarm.config.undertow.configuration.mod_cluster.balancer.Node;
import org.wildfly.swarm.config.undertow.configuration.mod_cluster.balancer.LoadBalancingGroupConsumer;
import org.wildfly.swarm.config.undertow.configuration.mod_cluster.balancer.LoadBalancingGroupSupplier;
import org.wildfly.swarm.config.undertow.configuration.mod_cluster.balancer.LoadBalancingGroup;
/**
 * Runtime representation of a mod_cluster balancer
 */
@Address("/subsystem=undertow/configuration=filter/mod-cluster=*/balancer=*")
@ResourceType("balancer")
public class Balancer<T extends Balancer<T>> {

	private String key;
	private PropertyChangeSupport pcs;
	private Integer maxAttempts;
	private Boolean stickySession;
	private String stickySessionCookie;
	private Boolean stickySessionForce;
	private String stickySessionPath;
	private Boolean stickySessionRemove;
	private Integer waitWorker;
	private BalancerResources subresources = new BalancerResources();

	public Balancer(String key) {
		this.key = key;
	}

	public String getKey() {
		return this.key;
	}

	/**
	 * Adds a property change listener
	 */
	public void addPropertyChangeListener(PropertyChangeListener listener) {
		if (null == this.pcs)
			this.pcs = new PropertyChangeSupport(this);
		this.pcs.addPropertyChangeListener(listener);
	}

	/**
	 * Removes a property change listener
	 */
	public void removePropertyChangeListener(PropertyChangeListener listener) {
		if (this.pcs != null)
			this.pcs.removePropertyChangeListener(listener);
	}

	/**
	 * The number of attempts to send the request to a backend server
	 */
	@ModelNodeBinding(detypedName = "max-attempts")
	public Integer maxAttempts() {
		return this.maxAttempts;
	}

	/**
	 * The number of attempts to send the request to a backend server
	 */
	@SuppressWarnings("unchecked")
	public T maxAttempts(Integer value) {
		Object oldValue = this.maxAttempts;
		this.maxAttempts = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("maxAttempts", oldValue, value);
		return (T) this;
	}

	/**
	 * If sticky sessions are enabled
	 */
	@ModelNodeBinding(detypedName = "sticky-session")
	public Boolean stickySession() {
		return this.stickySession;
	}

	/**
	 * If sticky sessions are enabled
	 */
	@SuppressWarnings("unchecked")
	public T stickySession(Boolean value) {
		Object oldValue = this.stickySession;
		this.stickySession = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("stickySession", oldValue, value);
		return (T) this;
	}

	/**
	 * The session cookie name
	 */
	@ModelNodeBinding(detypedName = "sticky-session-cookie")
	public String stickySessionCookie() {
		return this.stickySessionCookie;
	}

	/**
	 * The session cookie name
	 */
	@SuppressWarnings("unchecked")
	public T stickySessionCookie(String value) {
		Object oldValue = this.stickySessionCookie;
		this.stickySessionCookie = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("stickySessionCookie", oldValue, value);
		return (T) this;
	}

	/**
	 * If this is true then an error will be returned if the request cannot be
	 * routed to the sticky node, otherwise it will be routed to another node
	 */
	@ModelNodeBinding(detypedName = "sticky-session-force")
	public Boolean stickySessionForce() {
		return this.stickySessionForce;
	}

	/**
	 * If this is true then an error will be returned if the request cannot be
	 * routed to the sticky node, otherwise it will be routed to another node
	 */
	@SuppressWarnings("unchecked")
	public T stickySessionForce(Boolean value) {
		Object oldValue = this.stickySessionForce;
		this.stickySessionForce = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("stickySessionForce", oldValue, value);
		return (T) this;
	}

	/**
	 * The path of the sticky session cookie
	 */
	@ModelNodeBinding(detypedName = "sticky-session-path")
	public String stickySessionPath() {
		return this.stickySessionPath;
	}

	/**
	 * The path of the sticky session cookie
	 */
	@SuppressWarnings("unchecked")
	public T stickySessionPath(String value) {
		Object oldValue = this.stickySessionPath;
		this.stickySessionPath = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("stickySessionPath", oldValue, value);
		return (T) this;
	}

	/**
	 * Remove the session cookie if the request cannot be routed to the correct
	 * host
	 */
	@ModelNodeBinding(detypedName = "sticky-session-remove")
	public Boolean stickySessionRemove() {
		return this.stickySessionRemove;
	}

	/**
	 * Remove the session cookie if the request cannot be routed to the correct
	 * host
	 */
	@SuppressWarnings("unchecked")
	public T stickySessionRemove(Boolean value) {
		Object oldValue = this.stickySessionRemove;
		this.stickySessionRemove = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("stickySessionRemove", oldValue, value);
		return (T) this;
	}

	/**
	 * The number of seconds to wait for an available worker
	 */
	@ModelNodeBinding(detypedName = "wait-worker")
	public Integer waitWorker() {
		return this.waitWorker;
	}

	/**
	 * The number of seconds to wait for an available worker
	 */
	@SuppressWarnings("unchecked")
	public T waitWorker(Integer value) {
		Object oldValue = this.waitWorker;
		this.waitWorker = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("waitWorker", oldValue, value);
		return (T) this;
	}

	public BalancerResources subresources() {
		return this.subresources;
	}

	/**
	 * Add all Node objects to this subresource
	 * 
	 * @return this
	 * @param value
	 *            List of Node objects.
	 */
	@SuppressWarnings("unchecked")
	public T nodes(List<Node> value) {
		this.subresources.nodes = value;
		return (T) this;
	}

	/**
	 * Add the Node object to the list of subresources
	 * 
	 * @param value
	 *            The Node to add
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T node(Node value) {
		this.subresources.nodes.add(value);
		return (T) this;
	}

	/**
	 * Create and configure a Node object to the list of subresources
	 * 
	 * @param key
	 *            The key for the Node resource
	 * @param config
	 *            The NodeConsumer to use
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T node(String childKey, NodeConsumer consumer) {
		Node<?> child = new Node<>(childKey);
		if (consumer != null) {
			consumer.accept(child);
		}
		node(child);
		return (T) this;
	}

	/**
	 * Create and configure a Node object to the list of subresources
	 * 
	 * @param key
	 *            The key for the Node resource
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T node(String childKey) {
		node(childKey, null);
		return (T) this;
	}

	/**
	 * Install a supplied Node object to the list of subresources
	 */
	@SuppressWarnings("unchecked")
	public T node(NodeSupplier supplier) {
		node(supplier.get());
		return (T) this;
	}

	/**
	 * Add all LoadBalancingGroup objects to this subresource
	 * 
	 * @return this
	 * @param value
	 *            List of LoadBalancingGroup objects.
	 */
	@SuppressWarnings("unchecked")
	public T loadBalancingGroups(List<LoadBalancingGroup> value) {
		this.subresources.loadBalancingGroups = value;
		return (T) this;
	}

	/**
	 * Add the LoadBalancingGroup object to the list of subresources
	 * 
	 * @param value
	 *            The LoadBalancingGroup to add
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T loadBalancingGroup(LoadBalancingGroup value) {
		this.subresources.loadBalancingGroups.add(value);
		return (T) this;
	}

	/**
	 * Create and configure a LoadBalancingGroup object to the list of
	 * subresources
	 * 
	 * @param key
	 *            The key for the LoadBalancingGroup resource
	 * @param config
	 *            The LoadBalancingGroupConsumer to use
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T loadBalancingGroup(String childKey,
			LoadBalancingGroupConsumer consumer) {
		LoadBalancingGroup<?> child = new LoadBalancingGroup<>(childKey);
		if (consumer != null) {
			consumer.accept(child);
		}
		loadBalancingGroup(child);
		return (T) this;
	}

	/**
	 * Create and configure a LoadBalancingGroup object to the list of
	 * subresources
	 * 
	 * @param key
	 *            The key for the LoadBalancingGroup resource
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T loadBalancingGroup(String childKey) {
		loadBalancingGroup(childKey, null);
		return (T) this;
	}

	/**
	 * Install a supplied LoadBalancingGroup object to the list of subresources
	 */
	@SuppressWarnings("unchecked")
	public T loadBalancingGroup(LoadBalancingGroupSupplier supplier) {
		loadBalancingGroup(supplier.get());
		return (T) this;
	}

	/**
	 * Child mutators for Balancer
	 */
	public static class BalancerResources {
		/**
		 * Runtime representation of a mod_cluster node
		 */
		private List<Node> nodes = new java.util.ArrayList<>();
		/**
		 * A load balancing group
		 */
		private List<LoadBalancingGroup> loadBalancingGroups = new java.util.ArrayList<>();

		/**
		 * Get the list of Node resources
		 * 
		 * @return the list of resources
		 */
		@Subresource
		public List<Node> nodes() {
			return this.nodes;
		}

		/**
		 * Get the list of LoadBalancingGroup resources
		 * 
		 * @return the list of resources
		 */
		@Subresource
		public List<LoadBalancingGroup> loadBalancingGroups() {
			return this.loadBalancingGroups;
		}
	}
}