package org.wildfly.swarm.config;

import org.wildfly.swarm.config.runtime.Address;
import org.wildfly.swarm.config.runtime.ResourceType;
import org.wildfly.swarm.config.runtime.Implicit;
import java.beans.PropertyChangeSupport;
import java.beans.PropertyChangeListener;
import org.wildfly.swarm.config.runtime.ModelNodeBinding;
import java.util.List;
import org.wildfly.swarm.config.runtime.Subresource;
import org.wildfly.swarm.config.undertow.ServerConsumer;
import org.wildfly.swarm.config.undertow.ServerSupplier;
import org.wildfly.swarm.config.undertow.Server;
import org.wildfly.swarm.config.undertow.BufferCacheConsumer;
import org.wildfly.swarm.config.undertow.BufferCacheSupplier;
import org.wildfly.swarm.config.undertow.BufferCache;
import org.wildfly.swarm.config.undertow.ServletContainerConsumer;
import org.wildfly.swarm.config.undertow.ServletContainerSupplier;
import org.wildfly.swarm.config.undertow.ServletContainer;
import org.wildfly.swarm.config.undertow.HandlerConfiguration;
import org.wildfly.swarm.config.undertow.HandlerConfigurationConsumer;
import org.wildfly.swarm.config.undertow.HandlerConfigurationSupplier;
import org.wildfly.swarm.config.undertow.FilterConfiguration;
import org.wildfly.swarm.config.undertow.FilterConfigurationConsumer;
import org.wildfly.swarm.config.undertow.FilterConfigurationSupplier;
/**
 * Undertow subsystem
 */
@Address("/subsystem=undertow")
@ResourceType("subsystem")
@Implicit
public class Undertow<T extends Undertow<T>> {

	private String key;
	private PropertyChangeSupport pcs;
	private String defaultServer;
	private String defaultServletContainer;
	private String defaultVirtualHost;
	private String instanceId;
	private Boolean statisticsEnabled;
	private UndertowResources subresources = new UndertowResources();

	public Undertow() {
		this.key = "undertow";
		this.pcs = new PropertyChangeSupport(this);
	}

	public String getKey() {
		return this.key;
	}

	/**
	 * Adds a property change listener
	 */
	public void addPropertyChangeListener(PropertyChangeListener listener) {
		if (null == this.pcs)
			this.pcs = new PropertyChangeSupport(this);
		this.pcs.addPropertyChangeListener(listener);
	}

	/**
	 * Removes a property change listener
	 */
	public void removePropertyChangeListener(PropertyChangeListener listener) {
		if (this.pcs != null)
			this.pcs.removePropertyChangeListener(listener);
	}

	/**
	 * The default server to use for deployments
	 */
	@ModelNodeBinding(detypedName = "default-server")
	public String defaultServer() {
		return this.defaultServer;
	}

	/**
	 * The default server to use for deployments
	 */
	@SuppressWarnings("unchecked")
	public T defaultServer(String value) {
		Object oldValue = this.defaultServer;
		this.defaultServer = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("defaultServer", oldValue, value);
		return (T) this;
	}

	/**
	 * The default servlet container to use for deployments
	 */
	@ModelNodeBinding(detypedName = "default-servlet-container")
	public String defaultServletContainer() {
		return this.defaultServletContainer;
	}

	/**
	 * The default servlet container to use for deployments
	 */
	@SuppressWarnings("unchecked")
	public T defaultServletContainer(String value) {
		Object oldValue = this.defaultServletContainer;
		this.defaultServletContainer = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("defaultServletContainer", oldValue,
					value);
		return (T) this;
	}

	/**
	 * The default virtual host to use for deployments
	 */
	@ModelNodeBinding(detypedName = "default-virtual-host")
	public String defaultVirtualHost() {
		return this.defaultVirtualHost;
	}

	/**
	 * The default virtual host to use for deployments
	 */
	@SuppressWarnings("unchecked")
	public T defaultVirtualHost(String value) {
		Object oldValue = this.defaultVirtualHost;
		this.defaultVirtualHost = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("defaultVirtualHost", oldValue, value);
		return (T) this;
	}

	/**
	 * The cluster instance id
	 */
	@ModelNodeBinding(detypedName = "instance-id")
	public String instanceId() {
		return this.instanceId;
	}

	/**
	 * The cluster instance id
	 */
	@SuppressWarnings("unchecked")
	public T instanceId(String value) {
		Object oldValue = this.instanceId;
		this.instanceId = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("instanceId", oldValue, value);
		return (T) this;
	}

	/**
	 * Configures if are statistics enabled
	 */
	@ModelNodeBinding(detypedName = "statistics-enabled")
	public Boolean statisticsEnabled() {
		return this.statisticsEnabled;
	}

	/**
	 * Configures if are statistics enabled
	 */
	@SuppressWarnings("unchecked")
	public T statisticsEnabled(Boolean value) {
		Object oldValue = this.statisticsEnabled;
		this.statisticsEnabled = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("statisticsEnabled", oldValue, value);
		return (T) this;
	}

	public UndertowResources subresources() {
		return this.subresources;
	}

	/**
	 * Add all Server objects to this subresource
	 * 
	 * @return this
	 * @param value
	 *            List of Server objects.
	 */
	@SuppressWarnings("unchecked")
	public T servers(List<Server> value) {
		this.subresources.servers = value;
		return (T) this;
	}

	/**
	 * Add the Server object to the list of subresources
	 * 
	 * @param value
	 *            The Server to add
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T server(Server value) {
		this.subresources.servers.add(value);
		return (T) this;
	}

	/**
	 * Create and configure a Server object to the list of subresources
	 * 
	 * @param key
	 *            The key for the Server resource
	 * @param config
	 *            The ServerConsumer to use
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T server(String childKey, ServerConsumer consumer) {
		Server<? extends Server> child = new Server<>(childKey);
		if (consumer != null) {
			consumer.accept(child);
		}
		server(child);
		return (T) this;
	}

	/**
	 * Create and configure a Server object to the list of subresources
	 * 
	 * @param key
	 *            The key for the Server resource
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T server(String childKey) {
		server(childKey, null);
		return (T) this;
	}

	/**
	 * Install a supplied Server object to the list of subresources
	 */
	@SuppressWarnings("unchecked")
	public T server(ServerSupplier supplier) {
		server(supplier.get());
		return (T) this;
	}

	/**
	 * Add all BufferCache objects to this subresource
	 * 
	 * @return this
	 * @param value
	 *            List of BufferCache objects.
	 */
	@SuppressWarnings("unchecked")
	public T bufferCaches(List<BufferCache> value) {
		this.subresources.bufferCaches = value;
		return (T) this;
	}

	/**
	 * Add the BufferCache object to the list of subresources
	 * 
	 * @param value
	 *            The BufferCache to add
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T bufferCache(BufferCache value) {
		this.subresources.bufferCaches.add(value);
		return (T) this;
	}

	/**
	 * Create and configure a BufferCache object to the list of subresources
	 * 
	 * @param key
	 *            The key for the BufferCache resource
	 * @param config
	 *            The BufferCacheConsumer to use
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T bufferCache(String childKey, BufferCacheConsumer consumer) {
		BufferCache<? extends BufferCache> child = new BufferCache<>(childKey);
		if (consumer != null) {
			consumer.accept(child);
		}
		bufferCache(child);
		return (T) this;
	}

	/**
	 * Create and configure a BufferCache object to the list of subresources
	 * 
	 * @param key
	 *            The key for the BufferCache resource
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T bufferCache(String childKey) {
		bufferCache(childKey, null);
		return (T) this;
	}

	/**
	 * Install a supplied BufferCache object to the list of subresources
	 */
	@SuppressWarnings("unchecked")
	public T bufferCache(BufferCacheSupplier supplier) {
		bufferCache(supplier.get());
		return (T) this;
	}

	/**
	 * Add all ServletContainer objects to this subresource
	 * 
	 * @return this
	 * @param value
	 *            List of ServletContainer objects.
	 */
	@SuppressWarnings("unchecked")
	public T servletContainers(List<ServletContainer> value) {
		this.subresources.servletContainers = value;
		return (T) this;
	}

	/**
	 * Add the ServletContainer object to the list of subresources
	 * 
	 * @param value
	 *            The ServletContainer to add
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T servletContainer(ServletContainer value) {
		this.subresources.servletContainers.add(value);
		return (T) this;
	}

	/**
	 * Create and configure a ServletContainer object to the list of
	 * subresources
	 * 
	 * @param key
	 *            The key for the ServletContainer resource
	 * @param config
	 *            The ServletContainerConsumer to use
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T servletContainer(String childKey, ServletContainerConsumer consumer) {
		ServletContainer<? extends ServletContainer> child = new ServletContainer<>(
				childKey);
		if (consumer != null) {
			consumer.accept(child);
		}
		servletContainer(child);
		return (T) this;
	}

	/**
	 * Create and configure a ServletContainer object to the list of
	 * subresources
	 * 
	 * @param key
	 *            The key for the ServletContainer resource
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T servletContainer(String childKey) {
		servletContainer(childKey, null);
		return (T) this;
	}

	/**
	 * Install a supplied ServletContainer object to the list of subresources
	 */
	@SuppressWarnings("unchecked")
	public T servletContainer(ServletContainerSupplier supplier) {
		servletContainer(supplier.get());
		return (T) this;
	}

	/**
	 * Undertow handlers
	 */
	@SuppressWarnings("unchecked")
	public T handlerConfiguration(HandlerConfiguration value) {
		this.subresources.handlerConfiguration = value;
		return (T) this;
	}

	/**
	 * Undertow handlers
	 */
	@SuppressWarnings("unchecked")
	public T handlerConfiguration(HandlerConfigurationConsumer consumer) {
		HandlerConfiguration<? extends HandlerConfiguration> child = new HandlerConfiguration<>();
		if (consumer != null) {
			consumer.accept(child);
		}
		this.subresources.handlerConfiguration = child;
		return (T) this;
	}

	/**
	 * Undertow handlers
	 */
	@SuppressWarnings("unchecked")
	public T handlerConfiguration() {
		HandlerConfiguration<? extends HandlerConfiguration> child = new HandlerConfiguration<>();
		this.subresources.handlerConfiguration = child;
		return (T) this;
	}

	/**
	 * Undertow handlers
	 */
	@SuppressWarnings("unchecked")
	public T handlerConfiguration(HandlerConfigurationSupplier supplier) {
		this.subresources.handlerConfiguration = supplier.get();
		return (T) this;
	}

	/**
	 * Undertow filters
	 */
	@SuppressWarnings("unchecked")
	public T filterConfiguration(FilterConfiguration value) {
		this.subresources.filterConfiguration = value;
		return (T) this;
	}

	/**
	 * Undertow filters
	 */
	@SuppressWarnings("unchecked")
	public T filterConfiguration(FilterConfigurationConsumer consumer) {
		FilterConfiguration<? extends FilterConfiguration> child = new FilterConfiguration<>();
		if (consumer != null) {
			consumer.accept(child);
		}
		this.subresources.filterConfiguration = child;
		return (T) this;
	}

	/**
	 * Undertow filters
	 */
	@SuppressWarnings("unchecked")
	public T filterConfiguration() {
		FilterConfiguration<? extends FilterConfiguration> child = new FilterConfiguration<>();
		this.subresources.filterConfiguration = child;
		return (T) this;
	}

	/**
	 * Undertow filters
	 */
	@SuppressWarnings("unchecked")
	public T filterConfiguration(FilterConfigurationSupplier supplier) {
		this.subresources.filterConfiguration = supplier.get();
		return (T) this;
	}

	/**
	 * Child mutators for Undertow
	 */
	public static class UndertowResources {
		/**
		 * A server
		 */
		private List<Server> servers = new java.util.ArrayList<>();
		/**
		 * The buffer cache used to cache static content
		 */
		private List<BufferCache> bufferCaches = new java.util.ArrayList<>();
		/**
		 * A servlet container
		 */
		private List<ServletContainer> servletContainers = new java.util.ArrayList<>();
		private HandlerConfiguration handlerConfiguration;
		private FilterConfiguration filterConfiguration;

		/**
		 * Get the list of Server resources
		 * 
		 * @return the list of resources
		 */
		@Subresource
		public List<Server> servers() {
			return this.servers;
		}

		/**
		 * Get the list of BufferCache resources
		 * 
		 * @return the list of resources
		 */
		@Subresource
		public List<BufferCache> bufferCaches() {
			return this.bufferCaches;
		}

		/**
		 * Get the list of ServletContainer resources
		 * 
		 * @return the list of resources
		 */
		@Subresource
		public List<ServletContainer> servletContainers() {
			return this.servletContainers;
		}

		/**
		 * Undertow handlers
		 */
		@Subresource
		public HandlerConfiguration handlerConfiguration() {
			return this.handlerConfiguration;
		}

		/**
		 * Undertow filters
		 */
		@Subresource
		public FilterConfiguration filterConfiguration() {
			return this.filterConfiguration;
		}
	}
}