package org.wildfly.swarm.config.ejb3;

import org.wildfly.swarm.config.runtime.Address;
import org.wildfly.swarm.config.runtime.ResourceType;
import org.wildfly.swarm.config.runtime.Implicit;
import java.beans.PropertyChangeSupport;
import java.beans.PropertyChangeListener;
import org.wildfly.swarm.config.runtime.ModelNodeBinding;
import java.util.List;
import org.wildfly.swarm.config.runtime.Subresource;
import org.wildfly.swarm.config.ejb3.ChannelCreationOptionsConsumer;
import org.wildfly.swarm.config.ejb3.ChannelCreationOptionsSupplier;
import org.wildfly.swarm.config.ejb3.ChannelCreationOptions;
/**
 * The EJB3 Remote Service
 */
@Address("/subsystem=ejb3/service=remote")
@ResourceType("service")
@Implicit
public class RemoteService<T extends RemoteService<T>> {

	private String key;
	private PropertyChangeSupport pcs;
	private String cluster;
	private String connectorRef;
	private String threadPoolName;
	private RemoteServiceResources subresources = new RemoteServiceResources();

	public RemoteService() {
		this.key = "remote";
		this.pcs = new PropertyChangeSupport(this);
	}

	public String getKey() {
		return this.key;
	}

	/**
	 * Adds a property change listener
	 */
	public void addPropertyChangeListener(PropertyChangeListener listener) {
		if (null == this.pcs)
			this.pcs = new PropertyChangeSupport(this);
		this.pcs.addPropertyChangeListener(listener);
	}

	/**
	 * Removes a property change listener
	 */
	public void removePropertyChangeListener(PropertyChangeListener listener) {
		if (this.pcs != null)
			this.pcs.removePropertyChangeListener(listener);
	}

	/**
	 * The name of the clustered cache container which will be used to
	 * store/access the client-mappings of the EJB remoting connector's
	 * socket-binding on each node, in the cluster
	 */
	@ModelNodeBinding(detypedName = "cluster")
	public String cluster() {
		return this.cluster;
	}

	/**
	 * The name of the clustered cache container which will be used to
	 * store/access the client-mappings of the EJB remoting connector's
	 * socket-binding on each node, in the cluster
	 */
	@SuppressWarnings("unchecked")
	public T cluster(String value) {
		Object oldValue = this.cluster;
		this.cluster = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("cluster", oldValue, value);
		return (T) this;
	}

	/**
	 * The name of the connector on which the EJB3 remoting channel is
	 * registered
	 */
	@ModelNodeBinding(detypedName = "connector-ref")
	public String connectorRef() {
		return this.connectorRef;
	}

	/**
	 * The name of the connector on which the EJB3 remoting channel is
	 * registered
	 */
	@SuppressWarnings("unchecked")
	public T connectorRef(String value) {
		Object oldValue = this.connectorRef;
		this.connectorRef = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("connectorRef", oldValue, value);
		return (T) this;
	}

	/**
	 * The name of the thread pool that handles remote invocations
	 */
	@ModelNodeBinding(detypedName = "thread-pool-name")
	public String threadPoolName() {
		return this.threadPoolName;
	}

	/**
	 * The name of the thread pool that handles remote invocations
	 */
	@SuppressWarnings("unchecked")
	public T threadPoolName(String value) {
		Object oldValue = this.threadPoolName;
		this.threadPoolName = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("threadPoolName", oldValue, value);
		return (T) this;
	}

	public RemoteServiceResources subresources() {
		return this.subresources;
	}

	/**
	 * Add all ChannelCreationOptions objects to this subresource
	 * 
	 * @return this
	 * @param value
	 *            List of ChannelCreationOptions objects.
	 */
	@SuppressWarnings("unchecked")
	public T channelCreationOptions(List<ChannelCreationOptions> value) {
		this.subresources.channelCreationOptions = value;
		return (T) this;
	}

	/**
	 * Add the ChannelCreationOptions object to the list of subresources
	 * 
	 * @param value
	 *            The ChannelCreationOptions to add
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T channelCreationOptions(ChannelCreationOptions value) {
		this.subresources.channelCreationOptions.add(value);
		return (T) this;
	}

	/**
	 * Create and configure a ChannelCreationOptions object to the list of
	 * subresources
	 * 
	 * @param key
	 *            The key for the ChannelCreationOptions resource
	 * @param config
	 *            The ChannelCreationOptionsConsumer to use
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T channelCreationOptions(String childKey,
			ChannelCreationOptionsConsumer consumer) {
		ChannelCreationOptions<? extends ChannelCreationOptions> child = new ChannelCreationOptions<>(
				childKey);
		if (consumer != null) {
			consumer.accept(child);
		}
		channelCreationOptions(child);
		return (T) this;
	}

	/**
	 * Create and configure a ChannelCreationOptions object to the list of
	 * subresources
	 * 
	 * @param key
	 *            The key for the ChannelCreationOptions resource
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T channelCreationOptions(String childKey) {
		channelCreationOptions(childKey, null);
		return (T) this;
	}

	/**
	 * Install a supplied ChannelCreationOptions object to the list of
	 * subresources
	 */
	@SuppressWarnings("unchecked")
	public T channelCreationOptions(ChannelCreationOptionsSupplier supplier) {
		channelCreationOptions(supplier.get());
		return (T) this;
	}

	/**
	 * Child mutators for RemoteService
	 */
	public static class RemoteServiceResources {
		/**
		 * The options that will be used during the EJB remote channel creation
		 */
		private List<ChannelCreationOptions> channelCreationOptions = new java.util.ArrayList<>();

		/**
		 * Get the list of ChannelCreationOptions resources
		 * 
		 * @return the list of resources
		 */
		@Subresource
		public List<ChannelCreationOptions> channelCreationOptions() {
			return this.channelCreationOptions;
		}
	}
}