package org.wildfly.swarm.config.jgroups.stack.relay;

import org.wildfly.swarm.config.runtime.Address;
import org.wildfly.swarm.config.runtime.ResourceType;
import org.wildfly.swarm.config.runtime.Implicit;
import java.beans.PropertyChangeSupport;
import java.beans.PropertyChangeListener;
import org.wildfly.swarm.config.runtime.ModelNodeBinding;
import java.util.Map;
import java.util.List;
import org.wildfly.swarm.config.runtime.Subresource;
import org.wildfly.swarm.config.jgroups.PropertyConsumer;
import org.wildfly.swarm.config.jgroups.PropertySupplier;
import org.wildfly.swarm.config.jgroups.Property;
import org.wildfly.swarm.config.jgroups.stack.relay.RemoteSiteConsumer;
import org.wildfly.swarm.config.jgroups.stack.relay.RemoteSiteSupplier;
import org.wildfly.swarm.config.jgroups.stack.relay.RemoteSite;
/**
 * The configuration of a RELAY protocol
 */
@Address("/subsystem=jgroups/stack=*/relay=relay.RELAY2")
@ResourceType("relay")
@Implicit
public class RELAY2<T extends RELAY2<T>> {

	private String key;
	private PropertyChangeSupport pcs;
	private Map properties;
	private String site;
	private RELAY2Resources subresources = new RELAY2Resources();

	public RELAY2() {
		this.key = "relay.RELAY2";
		this.pcs = new PropertyChangeSupport(this);
	}

	public String getKey() {
		return this.key;
	}

	/**
	 * Adds a property change listener
	 */
	public void addPropertyChangeListener(PropertyChangeListener listener) {
		if (null == this.pcs)
			this.pcs = new PropertyChangeSupport(this);
		this.pcs.addPropertyChangeListener(listener);
	}

	/**
	 * Removes a property change listener
	 */
	public void removePropertyChangeListener(PropertyChangeListener listener) {
		if (this.pcs != null)
			this.pcs.removePropertyChangeListener(listener);
	}

	/**
	 * The properties of this protocol.
	 */
	@ModelNodeBinding(detypedName = "properties")
	public Map properties() {
		return this.properties;
	}

	/**
	 * The properties of this protocol.
	 */
	@SuppressWarnings("unchecked")
	public T properties(Map value) {
		Object oldValue = this.properties;
		this.properties = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("properties", oldValue, value);
		return (T) this;
	}

	/**
	 * The properties of this protocol.
	 */
	public T property(String key, Object value) {
		if (this.properties == null) {
			this.properties = new java.util.HashMap<>();
		}
		this.properties.put(key, value);
		return (T) this;
	}

	/**
	 * The name of the local site.
	 */
	@ModelNodeBinding(detypedName = "site")
	public String site() {
		return this.site;
	}

	/**
	 * The name of the local site.
	 */
	@SuppressWarnings("unchecked")
	public T site(String value) {
		Object oldValue = this.site;
		this.site = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("site", oldValue, value);
		return (T) this;
	}

	public RELAY2Resources subresources() {
		return this.subresources;
	}

	/**
	 * Add all Property objects to this subresource
	 * 
	 * @return this
	 * @param value
	 *            List of Property objects.
	 */
	@SuppressWarnings("unchecked")
	public T properties(List<Property> value) {
		this.subresources.properties = value;
		return (T) this;
	}

	/**
	 * Add the Property object to the list of subresources
	 * 
	 * @param value
	 *            The Property to add
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T property(Property value) {
		this.subresources.properties.add(value);
		return (T) this;
	}

	/**
	 * Create and configure a Property object to the list of subresources
	 * 
	 * @param key
	 *            The key for the Property resource
	 * @param config
	 *            The PropertyConsumer to use
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T property(String childKey, PropertyConsumer consumer) {
		Property<? extends Property> child = new Property<>(childKey);
		if (consumer != null) {
			consumer.accept(child);
		}
		property(child);
		return (T) this;
	}

	/**
	 * Create and configure a Property object to the list of subresources
	 * 
	 * @param key
	 *            The key for the Property resource
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T property(String childKey) {
		property(childKey, null);
		return (T) this;
	}

	/**
	 * Install a supplied Property object to the list of subresources
	 */
	@SuppressWarnings("unchecked")
	public T property(PropertySupplier supplier) {
		property(supplier.get());
		return (T) this;
	}

	/**
	 * Add all RemoteSite objects to this subresource
	 * 
	 * @return this
	 * @param value
	 *            List of RemoteSite objects.
	 */
	@SuppressWarnings("unchecked")
	public T remoteSites(List<RemoteSite> value) {
		this.subresources.remoteSites = value;
		return (T) this;
	}

	/**
	 * Add the RemoteSite object to the list of subresources
	 * 
	 * @param value
	 *            The RemoteSite to add
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T remoteSite(RemoteSite value) {
		this.subresources.remoteSites.add(value);
		return (T) this;
	}

	/**
	 * Create and configure a RemoteSite object to the list of subresources
	 * 
	 * @param key
	 *            The key for the RemoteSite resource
	 * @param config
	 *            The RemoteSiteConsumer to use
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T remoteSite(String childKey, RemoteSiteConsumer consumer) {
		RemoteSite<? extends RemoteSite> child = new RemoteSite<>(childKey);
		if (consumer != null) {
			consumer.accept(child);
		}
		remoteSite(child);
		return (T) this;
	}

	/**
	 * Create and configure a RemoteSite object to the list of subresources
	 * 
	 * @param key
	 *            The key for the RemoteSite resource
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T remoteSite(String childKey) {
		remoteSite(childKey, null);
		return (T) this;
	}

	/**
	 * Install a supplied RemoteSite object to the list of subresources
	 */
	@SuppressWarnings("unchecked")
	public T remoteSite(RemoteSiteSupplier supplier) {
		remoteSite(supplier.get());
		return (T) this;
	}

	/**
	 * Child mutators for RELAY2
	 */
	public static class RELAY2Resources {
		/**
		 * A protocol property with name and value.
		 */
		private List<Property> properties = new java.util.ArrayList<>();
		/**
		 * A remote site to which to bridge.
		 */
		private List<RemoteSite> remoteSites = new java.util.ArrayList<>();

		/**
		 * Get the list of Property resources
		 * 
		 * @return the list of resources
		 */
		@Subresource
		public List<Property> properties() {
			return this.properties;
		}

		/**
		 * Get the list of RemoteSite resources
		 * 
		 * @return the list of resources
		 */
		@Subresource
		public List<RemoteSite> remoteSites() {
			return this.remoteSites;
		}
	}
}