package org.wildfly.swarm.config.messaging_activemq.server;

import org.wildfly.swarm.config.runtime.Address;
import org.wildfly.swarm.config.runtime.ResourceType;
import java.beans.PropertyChangeSupport;
import java.beans.PropertyChangeListener;
import org.wildfly.swarm.config.runtime.ModelNodeBinding;
import java.util.List;
import java.util.Arrays;
import org.wildfly.swarm.config.runtime.Subresource;
import org.wildfly.swarm.config.messaging_activemq.server.RoleConsumer;
import org.wildfly.swarm.config.messaging_activemq.server.RoleSupplier;
import org.wildfly.swarm.config.messaging_activemq.server.Role;
/**
 * A runtime-only resource representing a ActiveMQ "address". Zero or more
 * queues can be bound to a single address. When a message is routed, it is
 * routed to the set of queues bound to the message's address.
 */
@Address("/subsystem=messaging-activemq/server=*/core-address=*")
@ResourceType("core-address")
public class CoreAddress<T extends CoreAddress<T>> {

	private String key;
	private PropertyChangeSupport pcs;
	private List<String> bindingNames;
	private Long numberOfBytesPerPage;
	private Integer numberOfPages;
	private List<String> queueNames;
	private CoreAddressResources subresources = new CoreAddressResources();

	public CoreAddress(String key) {
		this.key = key;
	}

	public String getKey() {
		return this.key;
	}

	/**
	 * Adds a property change listener
	 */
	public void addPropertyChangeListener(PropertyChangeListener listener) {
		if (null == this.pcs)
			this.pcs = new PropertyChangeSupport(this);
		this.pcs.addPropertyChangeListener(listener);
	}

	/**
	 * Removes a property change listener
	 */
	public void removePropertyChangeListener(PropertyChangeListener listener) {
		if (this.pcs != null)
			this.pcs.removePropertyChangeListener(listener);
	}

	/**
	 * The names of all bindings (both queues and diverts) bound to this
	 * address.
	 */
	@ModelNodeBinding(detypedName = "binding-names")
	public List<String> bindingNames() {
		return this.bindingNames;
	}

	/**
	 * The names of all bindings (both queues and diverts) bound to this
	 * address.
	 */
	@SuppressWarnings("unchecked")
	public T bindingNames(List<String> value) {
		Object oldValue = this.bindingNames;
		this.bindingNames = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("bindingNames", oldValue, value);
		return (T) this;
	}

	/**
	 * The names of all bindings (both queues and diverts) bound to this
	 * address.
	 */
	public T bindingName(String value) {
		if (this.bindingNames == null) {
			this.bindingNames = new java.util.ArrayList<>();
		}
		this.bindingNames.add(value);
		return (T) this;
	}

	/**
	 * The names of all bindings (both queues and diverts) bound to this
	 * address.
	 */
	@SuppressWarnings("unchecked")
	public T bindingNames(String... args) {
		bindingNames(Arrays.asList(args));
		return (T) this;
	}

	/**
	 * The number of bytes used by each page for this address.
	 */
	@ModelNodeBinding(detypedName = "number-of-bytes-per-page")
	public Long numberOfBytesPerPage() {
		return this.numberOfBytesPerPage;
	}

	/**
	 * The number of bytes used by each page for this address.
	 */
	@SuppressWarnings("unchecked")
	public T numberOfBytesPerPage(Long value) {
		Object oldValue = this.numberOfBytesPerPage;
		this.numberOfBytesPerPage = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("numberOfBytesPerPage", oldValue, value);
		return (T) this;
	}

	/**
	 * The number of pages used by this address.
	 */
	@ModelNodeBinding(detypedName = "number-of-pages")
	public Integer numberOfPages() {
		return this.numberOfPages;
	}

	/**
	 * The number of pages used by this address.
	 */
	@SuppressWarnings("unchecked")
	public T numberOfPages(Integer value) {
		Object oldValue = this.numberOfPages;
		this.numberOfPages = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("numberOfPages", oldValue, value);
		return (T) this;
	}

	/**
	 * The names of the queues associated with the address.
	 */
	@ModelNodeBinding(detypedName = "queue-names")
	public List<String> queueNames() {
		return this.queueNames;
	}

	/**
	 * The names of the queues associated with the address.
	 */
	@SuppressWarnings("unchecked")
	public T queueNames(List<String> value) {
		Object oldValue = this.queueNames;
		this.queueNames = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("queueNames", oldValue, value);
		return (T) this;
	}

	/**
	 * The names of the queues associated with the address.
	 */
	public T queueName(String value) {
		if (this.queueNames == null) {
			this.queueNames = new java.util.ArrayList<>();
		}
		this.queueNames.add(value);
		return (T) this;
	}

	/**
	 * The names of the queues associated with the address.
	 */
	@SuppressWarnings("unchecked")
	public T queueNames(String... args) {
		queueNames(Arrays.asList(args));
		return (T) this;
	}

	public CoreAddressResources subresources() {
		return this.subresources;
	}

	/**
	 * Add all Role objects to this subresource
	 * 
	 * @return this
	 * @param value
	 *            List of Role objects.
	 */
	@SuppressWarnings("unchecked")
	public T roles(List<Role> value) {
		this.subresources.roles = value;
		return (T) this;
	}

	/**
	 * Add the Role object to the list of subresources
	 * 
	 * @param value
	 *            The Role to add
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T role(Role value) {
		this.subresources.roles.add(value);
		return (T) this;
	}

	/**
	 * Create and configure a Role object to the list of subresources
	 * 
	 * @param key
	 *            The key for the Role resource
	 * @param config
	 *            The RoleConsumer to use
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T role(String childKey, RoleConsumer consumer) {
		Role<? extends Role> child = new Role<>(childKey);
		if (consumer != null) {
			consumer.accept(child);
		}
		role(child);
		return (T) this;
	}

	/**
	 * Create and configure a Role object to the list of subresources
	 * 
	 * @param key
	 *            The key for the Role resource
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T role(String childKey) {
		role(childKey, null);
		return (T) this;
	}

	/**
	 * Install a supplied Role object to the list of subresources
	 */
	@SuppressWarnings("unchecked")
	public T role(RoleSupplier supplier) {
		role(supplier.get());
		return (T) this;
	}

	/**
	 * Child mutators for CoreAddress
	 */
	public static class CoreAddressResources {
		/**
		 * A security role.
		 */
		private List<Role> roles = new java.util.ArrayList<>();

		/**
		 * Get the list of Role resources
		 * 
		 * @return the list of resources
		 */
		@Subresource
		public List<Role> roles() {
			return this.roles;
		}
	}
}