package org.wildfly.swarm.config.security.security_domain;

import org.wildfly.swarm.config.runtime.Address;
import org.wildfly.swarm.config.runtime.ResourceType;
import org.wildfly.swarm.config.runtime.Implicit;
import java.beans.PropertyChangeSupport;
import java.beans.PropertyChangeListener;
import org.wildfly.swarm.config.runtime.ModelNodeBinding;
import java.util.List;
import org.wildfly.swarm.config.runtime.Subresource;
import org.wildfly.swarm.config.security.security_domain.acl.ACLModuleConsumer;
import org.wildfly.swarm.config.security.security_domain.acl.ACLModuleSupplier;
import org.wildfly.swarm.config.security.security_domain.acl.ACLModule;
/**
 * Access control list configuration. Configures a list of ACL modules to be
 * used.
 */
@Address("/subsystem=security/security-domain=*/acl=classic")
@ResourceType("acl")
@Implicit
public class ClassicACL<T extends ClassicACL<T>> {

	private String key;
	private PropertyChangeSupport pcs;
	private ClassicACLResources subresources = new ClassicACLResources();

	public ClassicACL() {
		this.key = "classic";
		this.pcs = new PropertyChangeSupport(this);
	}

	public String getKey() {
		return this.key;
	}

	/**
	 * Adds a property change listener
	 */
	public void addPropertyChangeListener(PropertyChangeListener listener) {
		if (null == this.pcs)
			this.pcs = new PropertyChangeSupport(this);
		this.pcs.addPropertyChangeListener(listener);
	}

	/**
	 * Removes a property change listener
	 */
	public void removePropertyChangeListener(PropertyChangeListener listener) {
		if (this.pcs != null)
			this.pcs.removePropertyChangeListener(listener);
	}

	public ClassicACLResources subresources() {
		return this.subresources;
	}

	/**
	 * Add all ACLModule objects to this subresource
	 * 
	 * @return this
	 * @param value
	 *            List of ACLModule objects.
	 */
	@SuppressWarnings("unchecked")
	public T aclModules(List<ACLModule> value) {
		this.subresources.aclModules = value;
		return (T) this;
	}

	/**
	 * Add the ACLModule object to the list of subresources
	 * 
	 * @param value
	 *            The ACLModule to add
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T aclModule(ACLModule value) {
		this.subresources.aclModules.add(value);
		return (T) this;
	}

	/**
	 * Create and configure a ACLModule object to the list of subresources
	 * 
	 * @param key
	 *            The key for the ACLModule resource
	 * @param config
	 *            The ACLModuleConsumer to use
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T aclModule(String childKey, ACLModuleConsumer consumer) {
		ACLModule<? extends ACLModule> child = new ACLModule<>(childKey);
		if (consumer != null) {
			consumer.accept(child);
		}
		aclModule(child);
		return (T) this;
	}

	/**
	 * Create and configure a ACLModule object to the list of subresources
	 * 
	 * @param key
	 *            The key for the ACLModule resource
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T aclModule(String childKey) {
		aclModule(childKey, null);
		return (T) this;
	}

	/**
	 * Install a supplied ACLModule object to the list of subresources
	 */
	@SuppressWarnings("unchecked")
	public T aclModule(ACLModuleSupplier supplier) {
		aclModule(supplier.get());
		return (T) this;
	}

	/**
	 * Child mutators for ClassicACL
	 */
	public static class ClassicACLResources {
		/**
		 * List of authentication modules
		 */
		private List<ACLModule> aclModules = new java.util.ArrayList<>();

		/**
		 * Get the list of ACLModule resources
		 * 
		 * @return the list of resources
		 */
		@Subresource
		public List<ACLModule> aclModules() {
			return this.aclModules;
		}
	}
}