package org.wildfly.swarm.config.infinispan.cache_container;

import org.wildfly.swarm.config.runtime.Addresses;
import org.wildfly.swarm.config.runtime.ResourceType;
import org.wildfly.swarm.config.runtime.Implicit;
import java.beans.PropertyChangeSupport;
import java.beans.PropertyChangeListener;
import org.wildfly.swarm.config.runtime.ModelNodeBinding;
/**
 * The state transfer configuration for distributed and replicated caches.
 */
@Addresses({
		"/subsystem=infinispan/cache-container=*/replicated-cache=*/component=state-transfer",
		"/subsystem=infinispan/cache-container=*/distributed-cache=*/component=state-transfer"})
@ResourceType("component")
@Implicit
public class StateTransferComponent<T extends StateTransferComponent<T>> {

	private String key;
	private PropertyChangeSupport pcs;
	private Integer chunkSize;
	private Long timeout;

	public StateTransferComponent() {
		this.key = "state-transfer";
		this.pcs = new PropertyChangeSupport(this);
	}

	public String getKey() {
		return this.key;
	}

	/**
	 * Adds a property change listener
	 */
	public void addPropertyChangeListener(PropertyChangeListener listener) {
		if (null == this.pcs)
			this.pcs = new PropertyChangeSupport(this);
		this.pcs.addPropertyChangeListener(listener);
	}

	/**
	 * Removes a property change listener
	 */
	public void removePropertyChangeListener(PropertyChangeListener listener) {
		if (this.pcs != null)
			this.pcs.removePropertyChangeListener(listener);
	}

	/**
	 * The size, in bytes, in which to batch the transfer of cache entries.
	 */
	@ModelNodeBinding(detypedName = "chunk-size")
	public Integer chunkSize() {
		return this.chunkSize;
	}

	/**
	 * The size, in bytes, in which to batch the transfer of cache entries.
	 */
	@SuppressWarnings("unchecked")
	public T chunkSize(Integer value) {
		Object oldValue = this.chunkSize;
		this.chunkSize = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("chunkSize", oldValue, value);
		return (T) this;
	}

	/**
	 * The maximum amount of time (ms) to wait for state from neighboring
	 * caches, before throwing an exception and aborting startup.
	 */
	@ModelNodeBinding(detypedName = "timeout")
	public Long timeout() {
		return this.timeout;
	}

	/**
	 * The maximum amount of time (ms) to wait for state from neighboring
	 * caches, before throwing an exception and aborting startup.
	 */
	@SuppressWarnings("unchecked")
	public T timeout(Long value) {
		Object oldValue = this.timeout;
		this.timeout = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("timeout", oldValue, value);
		return (T) this;
	}
}