package org.wildfly.swarm.config.logging;

import org.wildfly.swarm.config.runtime.Addresses;
import org.wildfly.swarm.config.runtime.ResourceType;
import java.beans.PropertyChangeSupport;
import java.beans.PropertyChangeListener;
import org.wildfly.swarm.config.runtime.ModelNodeBinding;
import java.util.Map;
/**
 * A custom formatter to be used with handlers. Note that most log records are
 * formatted in the printf format. Formatters may require invocation of the
 * org.jboss.logmanager.ExtLogRecord#getFormattedMessage() for the message to be
 * properly formatted.
 */
@Addresses({"/subsystem=logging/logging-profile=*/custom-formatter=*",
		"/subsystem=logging/custom-formatter=*"})
@ResourceType("custom-formatter")
public class CustomFormatter<T extends CustomFormatter<T>> {

	private String key;
	private PropertyChangeSupport pcs;
	private String attributeClass;
	private String module;
	private Map properties;

	public CustomFormatter(String key) {
		this.key = key;
	}

	public String getKey() {
		return this.key;
	}

	/**
	 * Adds a property change listener
	 */
	public void addPropertyChangeListener(PropertyChangeListener listener) {
		if (null == this.pcs)
			this.pcs = new PropertyChangeSupport(this);
		this.pcs.addPropertyChangeListener(listener);
	}

	/**
	 * Removes a property change listener
	 */
	public void removePropertyChangeListener(PropertyChangeListener listener) {
		if (this.pcs != null)
			this.pcs.removePropertyChangeListener(listener);
	}

	/**
	 * The logging handler class to be used.
	 */
	@ModelNodeBinding(detypedName = "class")
	public String attributeClass() {
		return this.attributeClass;
	}

	/**
	 * The logging handler class to be used.
	 */
	@SuppressWarnings("unchecked")
	public T attributeClass(String value) {
		Object oldValue = this.attributeClass;
		this.attributeClass = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("attributeClass", oldValue, value);
		return (T) this;
	}

	/**
	 * The module that the logging handler depends on.
	 */
	@ModelNodeBinding(detypedName = "module")
	public String module() {
		return this.module;
	}

	/**
	 * The module that the logging handler depends on.
	 */
	@SuppressWarnings("unchecked")
	public T module(String value) {
		Object oldValue = this.module;
		this.module = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("module", oldValue, value);
		return (T) this;
	}

	/**
	 * Defines the properties used for the logging handler. All properties must
	 * be accessible via a setter method.
	 */
	@ModelNodeBinding(detypedName = "properties")
	public Map properties() {
		return this.properties;
	}

	/**
	 * Defines the properties used for the logging handler. All properties must
	 * be accessible via a setter method.
	 */
	@SuppressWarnings("unchecked")
	public T properties(Map value) {
		Object oldValue = this.properties;
		this.properties = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("properties", oldValue, value);
		return (T) this;
	}

	/**
	 * Defines the properties used for the logging handler. All properties must
	 * be accessible via a setter method.
	 */
	public T property(String key, Object value) {
		if (this.properties == null) {
			this.properties = new java.util.HashMap<>();
		}
		this.properties.put(key, value);
		return (T) this;
	}
}