package org.wildfly.swarm.config.management.access.syslog_handler.protocol;

import org.wildfly.swarm.config.runtime.Address;
import org.wildfly.swarm.config.runtime.ResourceType;
import org.wildfly.swarm.config.runtime.Implicit;
import java.beans.PropertyChangeSupport;
import java.beans.PropertyChangeListener;
import org.wildfly.swarm.config.runtime.ModelNodeBinding;
/**
 * Configuration for the truststore for the server certificate, if not signed by
 * an authority.
 */
@Address("/core-service=management/access=audit/syslog-handler=*/protocol=tls/authentication=truststore")
@ResourceType("authentication")
@Implicit
public class TruststoreAuthentication<T extends TruststoreAuthentication<T>> {

	private String key;
	private PropertyChangeSupport pcs;
	private String keystorePassword;
	private String keystorePath;
	private String keystoreRelativeTo;

	public TruststoreAuthentication() {
		this.key = "truststore";
		this.pcs = new PropertyChangeSupport(this);
	}

	public String getKey() {
		return this.key;
	}

	/**
	 * Adds a property change listener
	 */
	public void addPropertyChangeListener(PropertyChangeListener listener) {
		if (null == this.pcs)
			this.pcs = new PropertyChangeSupport(this);
		this.pcs.addPropertyChangeListener(listener);
	}

	/**
	 * Removes a property change listener
	 */
	public void removePropertyChangeListener(PropertyChangeListener listener) {
		if (this.pcs != null)
			this.pcs.removePropertyChangeListener(listener);
	}

	/**
	 * The password for the truststore.
	 */
	@ModelNodeBinding(detypedName = "keystore-password")
	public String keystorePassword() {
		return this.keystorePassword;
	}

	/**
	 * The password for the truststore.
	 */
	@SuppressWarnings("unchecked")
	public T keystorePassword(String value) {
		Object oldValue = this.keystorePassword;
		this.keystorePassword = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("keystorePassword", oldValue, value);
		return (T) this;
	}

	/**
	 * =The path of the truststore.
	 */
	@ModelNodeBinding(detypedName = "keystore-path")
	public String keystorePath() {
		return this.keystorePath;
	}

	/**
	 * =The path of the truststore.
	 */
	@SuppressWarnings("unchecked")
	public T keystorePath(String value) {
		Object oldValue = this.keystorePath;
		this.keystorePath = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("keystorePath", oldValue, value);
		return (T) this;
	}

	/**
	 * The name of another previously named path, or of one of the standard
	 * paths provided by the system. If 'keystore-relative-to' is provided, the
	 * value of the 'keystore-path' attribute is treated as relative to the path
	 * specified by this attribute.
	 */
	@ModelNodeBinding(detypedName = "keystore-relative-to")
	public String keystoreRelativeTo() {
		return this.keystoreRelativeTo;
	}

	/**
	 * The name of another previously named path, or of one of the standard
	 * paths provided by the system. If 'keystore-relative-to' is provided, the
	 * value of the 'keystore-path' attribute is treated as relative to the path
	 * specified by this attribute.
	 */
	@SuppressWarnings("unchecked")
	public T keystoreRelativeTo(String value) {
		Object oldValue = this.keystoreRelativeTo;
		this.keystoreRelativeTo = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("keystoreRelativeTo", oldValue, value);
		return (T) this;
	}
}