package org.wildfly.swarm.config.messaging_activemq.server;

import org.wildfly.swarm.config.runtime.Address;
import org.wildfly.swarm.config.runtime.ResourceType;
import java.beans.PropertyChangeSupport;
import java.beans.PropertyChangeListener;
import org.wildfly.swarm.config.runtime.ModelNodeBinding;
/**
 * An address setting defines some attributes that are defined against an
 * address wildcard rather than a specific queue.
 */
@Address("/subsystem=messaging-activemq/server=*/address-setting=*")
@ResourceType("address-setting")
public class AddressSetting<T extends AddressSetting<T>> {

	private String key;
	private PropertyChangeSupport pcs;
	private String addressFullPolicy;
	private String deadLetterAddress;
	private String expiryAddress;
	private Long expiryDelay;
	private Boolean lastValueQueue;
	private Integer maxDeliveryAttempts;
	private Long maxRedeliveryDelay;
	private Long maxSizeBytes;
	private Integer messageCounterHistoryDayLimit;
	private Integer pageMaxCacheSize;
	private Long pageSizeBytes;
	private Long redeliveryDelay;
	private Double redeliveryMultiplier;
	private Long redistributionDelay;
	private Boolean sendToDlaOnNoRoute;
	private Long slowConsumerCheckPeriod;
	private String slowConsumerPolicy;
	private Long slowConsumerThreshold;

	public AddressSetting(String key) {
		this.key = key;
	}

	public String getKey() {
		return this.key;
	}

	/**
	 * Adds a property change listener
	 */
	public void addPropertyChangeListener(PropertyChangeListener listener) {
		if (null == this.pcs)
			this.pcs = new PropertyChangeSupport(this);
		this.pcs.addPropertyChangeListener(listener);
	}

	/**
	 * Removes a property change listener
	 */
	public void removePropertyChangeListener(PropertyChangeListener listener) {
		if (this.pcs != null)
			this.pcs.removePropertyChangeListener(listener);
	}

	/**
	 * Determines what happens when an address where max-size-bytes is specified
	 * becomes full. (PAGE, DROP or BLOCK)
	 */
	@ModelNodeBinding(detypedName = "address-full-policy")
	public String addressFullPolicy() {
		return this.addressFullPolicy;
	}

	/**
	 * Determines what happens when an address where max-size-bytes is specified
	 * becomes full. (PAGE, DROP or BLOCK)
	 */
	@SuppressWarnings("unchecked")
	public T addressFullPolicy(String value) {
		Object oldValue = this.addressFullPolicy;
		this.addressFullPolicy = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("addressFullPolicy", oldValue, value);
		return (T) this;
	}

	/**
	 * The dead letter address
	 */
	@ModelNodeBinding(detypedName = "dead-letter-address")
	public String deadLetterAddress() {
		return this.deadLetterAddress;
	}

	/**
	 * The dead letter address
	 */
	@SuppressWarnings("unchecked")
	public T deadLetterAddress(String value) {
		Object oldValue = this.deadLetterAddress;
		this.deadLetterAddress = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("deadLetterAddress", oldValue, value);
		return (T) this;
	}

	/**
	 * Defines where to send a message that has expired.
	 */
	@ModelNodeBinding(detypedName = "expiry-address")
	public String expiryAddress() {
		return this.expiryAddress;
	}

	/**
	 * Defines where to send a message that has expired.
	 */
	@SuppressWarnings("unchecked")
	public T expiryAddress(String value) {
		Object oldValue = this.expiryAddress;
		this.expiryAddress = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("expiryAddress", oldValue, value);
		return (T) this;
	}

	/**
	 * Defines the expiration time that will be used for messages using the
	 * default expiration time
	 */
	@ModelNodeBinding(detypedName = "expiry-delay")
	public Long expiryDelay() {
		return this.expiryDelay;
	}

	/**
	 * Defines the expiration time that will be used for messages using the
	 * default expiration time
	 */
	@SuppressWarnings("unchecked")
	public T expiryDelay(Long value) {
		Object oldValue = this.expiryDelay;
		this.expiryDelay = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("expiryDelay", oldValue, value);
		return (T) this;
	}

	/**
	 * Defines whether a queue only uses last values or not
	 */
	@ModelNodeBinding(detypedName = "last-value-queue")
	public Boolean lastValueQueue() {
		return this.lastValueQueue;
	}

	/**
	 * Defines whether a queue only uses last values or not
	 */
	@SuppressWarnings("unchecked")
	public T lastValueQueue(Boolean value) {
		Object oldValue = this.lastValueQueue;
		this.lastValueQueue = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("lastValueQueue", oldValue, value);
		return (T) this;
	}

	/**
	 * Defines how many time a cancelled message can be redelivered before
	 * sending to the dead-letter-address
	 */
	@ModelNodeBinding(detypedName = "max-delivery-attempts")
	public Integer maxDeliveryAttempts() {
		return this.maxDeliveryAttempts;
	}

	/**
	 * Defines how many time a cancelled message can be redelivered before
	 * sending to the dead-letter-address
	 */
	@SuppressWarnings("unchecked")
	public T maxDeliveryAttempts(Integer value) {
		Object oldValue = this.maxDeliveryAttempts;
		this.maxDeliveryAttempts = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("maxDeliveryAttempts", oldValue, value);
		return (T) this;
	}

	/**
	 * Maximum value for the redelivery-delay (in ms).
	 */
	@ModelNodeBinding(detypedName = "max-redelivery-delay")
	public Long maxRedeliveryDelay() {
		return this.maxRedeliveryDelay;
	}

	/**
	 * Maximum value for the redelivery-delay (in ms).
	 */
	@SuppressWarnings("unchecked")
	public T maxRedeliveryDelay(Long value) {
		Object oldValue = this.maxRedeliveryDelay;
		this.maxRedeliveryDelay = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("maxRedeliveryDelay", oldValue, value);
		return (T) this;
	}

	/**
	 * The max bytes size.
	 */
	@ModelNodeBinding(detypedName = "max-size-bytes")
	public Long maxSizeBytes() {
		return this.maxSizeBytes;
	}

	/**
	 * The max bytes size.
	 */
	@SuppressWarnings("unchecked")
	public T maxSizeBytes(Long value) {
		Object oldValue = this.maxSizeBytes;
		this.maxSizeBytes = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("maxSizeBytes", oldValue, value);
		return (T) this;
	}

	/**
	 * Day limit for the message counter history.
	 */
	@ModelNodeBinding(detypedName = "message-counter-history-day-limit")
	public Integer messageCounterHistoryDayLimit() {
		return this.messageCounterHistoryDayLimit;
	}

	/**
	 * Day limit for the message counter history.
	 */
	@SuppressWarnings("unchecked")
	public T messageCounterHistoryDayLimit(Integer value) {
		Object oldValue = this.messageCounterHistoryDayLimit;
		this.messageCounterHistoryDayLimit = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("messageCounterHistoryDayLimit",
					oldValue, value);
		return (T) this;
	}

	/**
	 * The number of page files to keep in memory to optimize IO during paging
	 * navigation.
	 */
	@ModelNodeBinding(detypedName = "page-max-cache-size")
	public Integer pageMaxCacheSize() {
		return this.pageMaxCacheSize;
	}

	/**
	 * The number of page files to keep in memory to optimize IO during paging
	 * navigation.
	 */
	@SuppressWarnings("unchecked")
	public T pageMaxCacheSize(Integer value) {
		Object oldValue = this.pageMaxCacheSize;
		this.pageMaxCacheSize = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("pageMaxCacheSize", oldValue, value);
		return (T) this;
	}

	/**
	 * The paging size.
	 */
	@ModelNodeBinding(detypedName = "page-size-bytes")
	public Long pageSizeBytes() {
		return this.pageSizeBytes;
	}

	/**
	 * The paging size.
	 */
	@SuppressWarnings("unchecked")
	public T pageSizeBytes(Long value) {
		Object oldValue = this.pageSizeBytes;
		this.pageSizeBytes = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("pageSizeBytes", oldValue, value);
		return (T) this;
	}

	/**
	 * Defines how long to wait before attempting redelivery of a cancelled
	 * message
	 */
	@ModelNodeBinding(detypedName = "redelivery-delay")
	public Long redeliveryDelay() {
		return this.redeliveryDelay;
	}

	/**
	 * Defines how long to wait before attempting redelivery of a cancelled
	 * message
	 */
	@SuppressWarnings("unchecked")
	public T redeliveryDelay(Long value) {
		Object oldValue = this.redeliveryDelay;
		this.redeliveryDelay = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("redeliveryDelay", oldValue, value);
		return (T) this;
	}

	/**
	 * Multiplier to apply to the redelivery-delay parameter
	 */
	@ModelNodeBinding(detypedName = "redelivery-multiplier")
	public Double redeliveryMultiplier() {
		return this.redeliveryMultiplier;
	}

	/**
	 * Multiplier to apply to the redelivery-delay parameter
	 */
	@SuppressWarnings("unchecked")
	public T redeliveryMultiplier(Double value) {
		Object oldValue = this.redeliveryMultiplier;
		this.redeliveryMultiplier = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("redeliveryMultiplier", oldValue, value);
		return (T) this;
	}

	/**
	 * Defines how long to wait when the last consumer is closed on a queue
	 * before redistributing any messages
	 */
	@ModelNodeBinding(detypedName = "redistribution-delay")
	public Long redistributionDelay() {
		return this.redistributionDelay;
	}

	/**
	 * Defines how long to wait when the last consumer is closed on a queue
	 * before redistributing any messages
	 */
	@SuppressWarnings("unchecked")
	public T redistributionDelay(Long value) {
		Object oldValue = this.redistributionDelay;
		this.redistributionDelay = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("redistributionDelay", oldValue, value);
		return (T) this;
	}

	/**
	 * If this parameter is set to true for that address, if the message is not
	 * routed to any queues it will instead be sent to the dead letter address
	 * (DLA) for that address, if it exists.
	 */
	@ModelNodeBinding(detypedName = "send-to-dla-on-no-route")
	public Boolean sendToDlaOnNoRoute() {
		return this.sendToDlaOnNoRoute;
	}

	/**
	 * If this parameter is set to true for that address, if the message is not
	 * routed to any queues it will instead be sent to the dead letter address
	 * (DLA) for that address, if it exists.
	 */
	@SuppressWarnings("unchecked")
	public T sendToDlaOnNoRoute(Boolean value) {
		Object oldValue = this.sendToDlaOnNoRoute;
		this.sendToDlaOnNoRoute = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("sendToDlaOnNoRoute", oldValue, value);
		return (T) this;
	}

	/**
	 * How often to check for slow consumers on a particular queue.
	 */
	@ModelNodeBinding(detypedName = "slow-consumer-check-period")
	public Long slowConsumerCheckPeriod() {
		return this.slowConsumerCheckPeriod;
	}

	/**
	 * How often to check for slow consumers on a particular queue.
	 */
	@SuppressWarnings("unchecked")
	public T slowConsumerCheckPeriod(Long value) {
		Object oldValue = this.slowConsumerCheckPeriod;
		this.slowConsumerCheckPeriod = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("slowConsumerCheckPeriod", oldValue,
					value);
		return (T) this;
	}

	/**
	 * Determine what happens when a slow consumer is identified.
	 */
	@ModelNodeBinding(detypedName = "slow-consumer-policy")
	public String slowConsumerPolicy() {
		return this.slowConsumerPolicy;
	}

	/**
	 * Determine what happens when a slow consumer is identified.
	 */
	@SuppressWarnings("unchecked")
	public T slowConsumerPolicy(String value) {
		Object oldValue = this.slowConsumerPolicy;
		this.slowConsumerPolicy = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("slowConsumerPolicy", oldValue, value);
		return (T) this;
	}

	/**
	 * The minimum rate of message consumption allowed before a consumer is
	 * considered slow.
	 */
	@ModelNodeBinding(detypedName = "slow-consumer-threshold")
	public Long slowConsumerThreshold() {
		return this.slowConsumerThreshold;
	}

	/**
	 * The minimum rate of message consumption allowed before a consumer is
	 * considered slow.
	 */
	@SuppressWarnings("unchecked")
	public T slowConsumerThreshold(Long value) {
		Object oldValue = this.slowConsumerThreshold;
		this.slowConsumerThreshold = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("slowConsumerThreshold", oldValue,
					value);
		return (T) this;
	}
}