package org.wildfly.swarm.config.messaging_activemq.server.ha_policy.replication_colocated;

import org.wildfly.swarm.config.runtime.Address;
import org.wildfly.swarm.config.runtime.ResourceType;
import org.wildfly.swarm.config.runtime.Implicit;
import java.beans.PropertyChangeSupport;
import java.beans.PropertyChangeListener;
import org.wildfly.swarm.config.runtime.ModelNodeBinding;
/**
 * A messaging resource that allows you to configure High Availability for the
 * ActiveMQ server (the value of ha-policy can be live-only, replication-master,
 * replication-slave, or replication-colocated).
 */
@Address("/subsystem=messaging-activemq/server=*/ha-policy=replication-colocated/configuration=master")
@ResourceType("configuration")
@Implicit
public class MasterConfiguration<T extends MasterConfiguration<T>> {

	private String key;
	private PropertyChangeSupport pcs;
	private Boolean checkForLiveServer;
	private String clusterName;
	private String groupName;

	public MasterConfiguration() {
		this.key = "master";
		this.pcs = new PropertyChangeSupport(this);
	}

	public String getKey() {
		return this.key;
	}

	/**
	 * Adds a property change listener
	 */
	public void addPropertyChangeListener(PropertyChangeListener listener) {
		if (null == this.pcs)
			this.pcs = new PropertyChangeSupport(this);
		this.pcs.addPropertyChangeListener(listener);
	}

	/**
	 * Removes a property change listener
	 */
	public void removePropertyChangeListener(PropertyChangeListener listener) {
		if (this.pcs != null)
			this.pcs.removePropertyChangeListener(listener);
	}

	/**
	 * Whether to check the cluster for another server using the same server ID
	 * when starting up.
	 */
	@ModelNodeBinding(detypedName = "check-for-live-server")
	public Boolean checkForLiveServer() {
		return this.checkForLiveServer;
	}

	/**
	 * Whether to check the cluster for another server using the same server ID
	 * when starting up.
	 */
	@SuppressWarnings("unchecked")
	public T checkForLiveServer(Boolean value) {
		Object oldValue = this.checkForLiveServer;
		this.checkForLiveServer = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("checkForLiveServer", oldValue, value);
		return (T) this;
	}

	/**
	 * Name of the cluster used for replication.
	 */
	@ModelNodeBinding(detypedName = "cluster-name")
	public String clusterName() {
		return this.clusterName;
	}

	/**
	 * Name of the cluster used for replication.
	 */
	@SuppressWarnings("unchecked")
	public T clusterName(String value) {
		Object oldValue = this.clusterName;
		this.clusterName = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("clusterName", oldValue, value);
		return (T) this;
	}

	/**
	 * If set, backup servers will only pair with live servers with matching
	 * group-name.
	 */
	@ModelNodeBinding(detypedName = "group-name")
	public String groupName() {
		return this.groupName;
	}

	/**
	 * If set, backup servers will only pair with live servers with matching
	 * group-name.
	 */
	@SuppressWarnings("unchecked")
	public T groupName(String value) {
		Object oldValue = this.groupName;
		this.groupName = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("groupName", oldValue, value);
		return (T) this;
	}
}