package org.wildfly.swarm.config.undertow.server;

import org.wildfly.swarm.config.runtime.Address;
import org.wildfly.swarm.config.runtime.ResourceType;
import java.beans.PropertyChangeSupport;
import java.beans.PropertyChangeListener;
import org.wildfly.swarm.config.runtime.ModelNodeBinding;
import java.util.List;
import java.util.Arrays;
import org.wildfly.swarm.config.runtime.Subresource;
import org.wildfly.swarm.config.undertow.server.host.FilterRefConsumer;
import org.wildfly.swarm.config.undertow.server.host.FilterRefSupplier;
import org.wildfly.swarm.config.undertow.server.host.FilterRef;
import org.wildfly.swarm.config.undertow.server.host.LocationConsumer;
import org.wildfly.swarm.config.undertow.server.host.LocationSupplier;
import org.wildfly.swarm.config.undertow.server.host.Location;
import org.wildfly.swarm.config.undertow.server.host.AccessLogSetting;
import org.wildfly.swarm.config.undertow.server.host.AccessLogSettingConsumer;
import org.wildfly.swarm.config.undertow.server.host.AccessLogSettingSupplier;
import org.wildfly.swarm.config.undertow.server.host.SingleSignOnSetting;
import org.wildfly.swarm.config.undertow.server.host.SingleSignOnSettingConsumer;
import org.wildfly.swarm.config.undertow.server.host.SingleSignOnSettingSupplier;
/**
 * An Undertow host
 */
@Address("/subsystem=undertow/server=*/host=*")
@ResourceType("host")
public class Host<T extends Host<T>> {

	private String key;
	private PropertyChangeSupport pcs;
	private List<String> alias;
	private Integer defaultResponseCode;
	private String defaultWebModule;
	private Boolean disableConsoleRedirect;
	private HostResources subresources = new HostResources();

	public Host(String key) {
		this.key = key;
	}

	public String getKey() {
		return this.key;
	}

	/**
	 * Adds a property change listener
	 */
	public void addPropertyChangeListener(PropertyChangeListener listener) {
		if (null == this.pcs)
			this.pcs = new PropertyChangeSupport(this);
		this.pcs.addPropertyChangeListener(listener);
	}

	/**
	 * Removes a property change listener
	 */
	public void removePropertyChangeListener(PropertyChangeListener listener) {
		if (this.pcs != null)
			this.pcs.removePropertyChangeListener(listener);
	}

	/**
	 * Aliases for the host
	 */
	@ModelNodeBinding(detypedName = "alias")
	public List<String> alias() {
		return this.alias;
	}

	/**
	 * Aliases for the host
	 */
	@SuppressWarnings("unchecked")
	public T alias(List<String> value) {
		Object oldValue = this.alias;
		this.alias = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("alias", oldValue, value);
		return (T) this;
	}

	/**
	 * Aliases for the host
	 */
	public T alias(String value) {
		if (this.alias == null) {
			this.alias = new java.util.ArrayList<>();
		}
		this.alias.add(value);
		return (T) this;
	}

	/**
	 * Aliases for the host
	 */
	@SuppressWarnings("unchecked")
	public T alias(String... args) {
		alias(Arrays.asList(args));
		return (T) this;
	}

	/**
	 * If set, this will be response code sent back in case requested context
	 * does not exist on server.
	 */
	@ModelNodeBinding(detypedName = "default-response-code")
	public Integer defaultResponseCode() {
		return this.defaultResponseCode;
	}

	/**
	 * If set, this will be response code sent back in case requested context
	 * does not exist on server.
	 */
	@SuppressWarnings("unchecked")
	public T defaultResponseCode(Integer value) {
		Object oldValue = this.defaultResponseCode;
		this.defaultResponseCode = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("defaultResponseCode", oldValue, value);
		return (T) this;
	}

	/**
	 * Default web module
	 */
	@ModelNodeBinding(detypedName = "default-web-module")
	public String defaultWebModule() {
		return this.defaultWebModule;
	}

	/**
	 * Default web module
	 */
	@SuppressWarnings("unchecked")
	public T defaultWebModule(String value) {
		Object oldValue = this.defaultWebModule;
		this.defaultWebModule = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("defaultWebModule", oldValue, value);
		return (T) this;
	}

	/**
	 * if set to true, /console redirect wont be enabled for this host, default
	 * is false
	 */
	@ModelNodeBinding(detypedName = "disable-console-redirect")
	public Boolean disableConsoleRedirect() {
		return this.disableConsoleRedirect;
	}

	/**
	 * if set to true, /console redirect wont be enabled for this host, default
	 * is false
	 */
	@SuppressWarnings("unchecked")
	public T disableConsoleRedirect(Boolean value) {
		Object oldValue = this.disableConsoleRedirect;
		this.disableConsoleRedirect = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("disableConsoleRedirect", oldValue,
					value);
		return (T) this;
	}

	public HostResources subresources() {
		return this.subresources;
	}

	/**
	 * Add all FilterRef objects to this subresource
	 * 
	 * @return this
	 * @param value
	 *            List of FilterRef objects.
	 */
	@SuppressWarnings("unchecked")
	public T filterRefs(List<FilterRef> value) {
		this.subresources.filterRefs = value;
		return (T) this;
	}

	/**
	 * Add the FilterRef object to the list of subresources
	 * 
	 * @param value
	 *            The FilterRef to add
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T filterRef(FilterRef value) {
		this.subresources.filterRefs.add(value);
		return (T) this;
	}

	/**
	 * Create and configure a FilterRef object to the list of subresources
	 * 
	 * @param key
	 *            The key for the FilterRef resource
	 * @param config
	 *            The FilterRefConsumer to use
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T filterRef(String childKey, FilterRefConsumer consumer) {
		FilterRef<? extends FilterRef> child = new FilterRef<>(childKey);
		if (consumer != null) {
			consumer.accept(child);
		}
		filterRef(child);
		return (T) this;
	}

	/**
	 * Create and configure a FilterRef object to the list of subresources
	 * 
	 * @param key
	 *            The key for the FilterRef resource
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T filterRef(String childKey) {
		filterRef(childKey, null);
		return (T) this;
	}

	/**
	 * Install a supplied FilterRef object to the list of subresources
	 */
	@SuppressWarnings("unchecked")
	public T filterRef(FilterRefSupplier supplier) {
		filterRef(supplier.get());
		return (T) this;
	}

	/**
	 * Add all Location objects to this subresource
	 * 
	 * @return this
	 * @param value
	 *            List of Location objects.
	 */
	@SuppressWarnings("unchecked")
	public T locations(List<Location> value) {
		this.subresources.locations = value;
		return (T) this;
	}

	/**
	 * Add the Location object to the list of subresources
	 * 
	 * @param value
	 *            The Location to add
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T location(Location value) {
		this.subresources.locations.add(value);
		return (T) this;
	}

	/**
	 * Create and configure a Location object to the list of subresources
	 * 
	 * @param key
	 *            The key for the Location resource
	 * @param config
	 *            The LocationConsumer to use
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T location(String childKey, LocationConsumer consumer) {
		Location<? extends Location> child = new Location<>(childKey);
		if (consumer != null) {
			consumer.accept(child);
		}
		location(child);
		return (T) this;
	}

	/**
	 * Create and configure a Location object to the list of subresources
	 * 
	 * @param key
	 *            The key for the Location resource
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T location(String childKey) {
		location(childKey, null);
		return (T) this;
	}

	/**
	 * Install a supplied Location object to the list of subresources
	 */
	@SuppressWarnings("unchecked")
	public T location(LocationSupplier supplier) {
		location(supplier.get());
		return (T) this;
	}

	/**
	 * The access log configuration for this virtual server.
	 */
	@SuppressWarnings("unchecked")
	public T accessLogSetting(AccessLogSetting value) {
		this.subresources.accessLogSetting = value;
		return (T) this;
	}

	/**
	 * The access log configuration for this virtual server.
	 */
	@SuppressWarnings("unchecked")
	public T accessLogSetting(AccessLogSettingConsumer consumer) {
		AccessLogSetting<? extends AccessLogSetting> child = new AccessLogSetting<>();
		if (consumer != null) {
			consumer.accept(child);
		}
		this.subresources.accessLogSetting = child;
		return (T) this;
	}

	/**
	 * The access log configuration for this virtual server.
	 */
	@SuppressWarnings("unchecked")
	public T accessLogSetting() {
		AccessLogSetting<? extends AccessLogSetting> child = new AccessLogSetting<>();
		this.subresources.accessLogSetting = child;
		return (T) this;
	}

	/**
	 * The access log configuration for this virtual server.
	 */
	@SuppressWarnings("unchecked")
	public T accessLogSetting(AccessLogSettingSupplier supplier) {
		this.subresources.accessLogSetting = supplier.get();
		return (T) this;
	}

	/**
	 * The SSO configuration for this virtual server.
	 */
	@SuppressWarnings("unchecked")
	public T singleSignOnSetting(SingleSignOnSetting value) {
		this.subresources.singleSignOnSetting = value;
		return (T) this;
	}

	/**
	 * The SSO configuration for this virtual server.
	 */
	@SuppressWarnings("unchecked")
	public T singleSignOnSetting(SingleSignOnSettingConsumer consumer) {
		SingleSignOnSetting<? extends SingleSignOnSetting> child = new SingleSignOnSetting<>();
		if (consumer != null) {
			consumer.accept(child);
		}
		this.subresources.singleSignOnSetting = child;
		return (T) this;
	}

	/**
	 * The SSO configuration for this virtual server.
	 */
	@SuppressWarnings("unchecked")
	public T singleSignOnSetting() {
		SingleSignOnSetting<? extends SingleSignOnSetting> child = new SingleSignOnSetting<>();
		this.subresources.singleSignOnSetting = child;
		return (T) this;
	}

	/**
	 * The SSO configuration for this virtual server.
	 */
	@SuppressWarnings("unchecked")
	public T singleSignOnSetting(SingleSignOnSettingSupplier supplier) {
		this.subresources.singleSignOnSetting = supplier.get();
		return (T) this;
	}

	/**
	 * Child mutators for Host
	 */
	public static class HostResources {
		/**
		 * Reference to filter
		 */
		private List<FilterRef> filterRefs = new java.util.ArrayList<>();
		/**
		 * Holds configuration of location resource
		 */
		private List<Location> locations = new java.util.ArrayList<>();
		private AccessLogSetting accessLogSetting;
		private SingleSignOnSetting singleSignOnSetting;

		/**
		 * Get the list of FilterRef resources
		 * 
		 * @return the list of resources
		 */
		@Subresource
		public List<FilterRef> filterRefs() {
			return this.filterRefs;
		}

		public FilterRef filterRef(String key) {
			return this.filterRefs.stream().filter(e -> e.getKey().equals(key))
					.findFirst().orElse(null);
		}
		/**
		 * Get the list of Location resources
		 * 
		 * @return the list of resources
		 */
		@Subresource
		public List<Location> locations() {
			return this.locations;
		}

		public Location location(String key) {
			return this.locations.stream().filter(e -> e.getKey().equals(key))
					.findFirst().orElse(null);
		}
		/**
		 * The access log configuration for this virtual server.
		 */
		@Subresource
		public AccessLogSetting accessLogSetting() {
			return this.accessLogSetting;
		}

		/**
		 * The SSO configuration for this virtual server.
		 */
		@Subresource
		public SingleSignOnSetting singleSignOnSetting() {
			return this.singleSignOnSetting;
		}
	}
}