package org.wildfly.swarm.config;

import org.wildfly.swarm.config.runtime.Address;
import org.wildfly.swarm.config.runtime.ResourceType;
import org.wildfly.swarm.config.runtime.Implicit;
import java.beans.PropertyChangeSupport;
import java.beans.PropertyChangeListener;
import org.wildfly.swarm.config.runtime.ModelNodeBinding;
import java.util.Map;
import java.util.Arrays;
/**
 * The iiop-openjdk subsystem configuration.
 */
@Address("/subsystem=iiop-openjdk")
@ResourceType("subsystem")
@Implicit
public class IIOPOpenjdk<T extends IIOPOpenjdk<T>> {

	private String key;
	private PropertyChangeSupport pcs;
	private Map properties;
	private AuthMethod authMethod;
	private String realm;
	private Boolean required;
	private Security security;
	private Transactions transactions;
	private Boolean exportCorbaloc;
	private String rootContext;
	private String giopVersion;
	private String persistentServerId;
	private String socketBinding;
	private String sslSocketBinding;
	private CallerPropagation callerPropagation;
	private Boolean addComponentViaInterceptor;
	private ClientRequires clientRequires;
	private ClientSupports clientSupports;
	private String securityDomain;
	private ServerRequires serverRequires;
	private ServerSupports serverSupports;
	private Boolean supportSsl;
	private Integer highWaterMark;
	private Integer numberToReclaim;
	private Confidentiality confidentiality;
	private DetectMisordering detectMisordering;
	private DetectReplay detectReplay;
	private Integrity integrity;
	private TrustInClient trustInClient;
	private TrustInTarget trustInTarget;

	public IIOPOpenjdk() {
		this.key = "iiop-openjdk";
		this.pcs = new PropertyChangeSupport(this);
	}

	public String getKey() {
		return this.key;
	}

	/**
	 * Adds a property change listener
	 */
	public void addPropertyChangeListener(PropertyChangeListener listener) {
		if (null == this.pcs)
			this.pcs = new PropertyChangeSupport(this);
		this.pcs.addPropertyChangeListener(listener);
	}

	/**
	 * Removes a property change listener
	 */
	public void removePropertyChangeListener(PropertyChangeListener listener) {
		if (this.pcs != null)
			this.pcs.removePropertyChangeListener(listener);
	}

	/**
	 * A list of generic key/value properties.
	 */
	@ModelNodeBinding(detypedName = "properties")
	public Map properties() {
		return this.properties;
	}

	/**
	 * A list of generic key/value properties.
	 */
	@SuppressWarnings("unchecked")
	public T properties(Map value) {
		Object oldValue = this.properties;
		this.properties = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("properties", oldValue, value);
		return (T) this;
	}

	/**
	 * A list of generic key/value properties.
	 */
	public T property(String key, Object value) {
		if (this.properties == null) {
			this.properties = new java.util.HashMap<>();
		}
		this.properties.put(key, value);
		return (T) this;
	}

	@Deprecated
	@SuppressWarnings("unchecked")
	public T authMethod(final String authMethod) {
		if (authMethod == null) {
			this.authMethod = null;
		} else {
			boolean found = false;
			for (AuthMethod e : AuthMethod.values()) {
				if (e.toString().equals(authMethod)) {
					authMethod(e);
					found = true;
					break;
				}
			}
			if (!found)
				throw new RuntimeException(String.format(
						"Value '%s' not valid. Valid values are: %s",
						authMethod, Arrays.asList(AuthMethod.values())));
		}
		return (T) this;
	}

	/**
	 * The authentication method. Valid values are 'none' and
	 * 'username_password'.
	 */
	@ModelNodeBinding(detypedName = "auth-method")
	public AuthMethod authMethod() {
		return this.authMethod;
	}

	/**
	 * The authentication method. Valid values are 'none' and
	 * 'username_password'.
	 */
	@SuppressWarnings("unchecked")
	public T authMethod(AuthMethod value) {
		Object oldValue = this.authMethod;
		this.authMethod = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("authMethod", oldValue, value);
		return (T) this;
	}

	/**
	 * The authentication service (AS) realm name.
	 */
	@ModelNodeBinding(detypedName = "realm")
	public String realm() {
		return this.realm;
	}

	/**
	 * The authentication service (AS) realm name.
	 */
	@SuppressWarnings("unchecked")
	public T realm(String value) {
		Object oldValue = this.realm;
		this.realm = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("realm", oldValue, value);
		return (T) this;
	}

	/**
	 * Indicates if authentication is required (true) or not (false).
	 */
	@ModelNodeBinding(detypedName = "required")
	public Boolean required() {
		return this.required;
	}

	/**
	 * Indicates if authentication is required (true) or not (false).
	 */
	@SuppressWarnings("unchecked")
	public T required(Boolean value) {
		Object oldValue = this.required;
		this.required = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("required", oldValue, value);
		return (T) this;
	}

	@Deprecated
	@SuppressWarnings("unchecked")
	public T security(final String security) {
		if (security == null) {
			this.security = null;
		} else {
			boolean found = false;
			for (Security e : Security.values()) {
				if (e.toString().equals(security)) {
					security(e);
					found = true;
					break;
				}
			}
			if (!found)
				throw new RuntimeException(String.format(
						"Value '%s' not valid. Valid values are: %s", security,
						Arrays.asList(Security.values())));
		}
		return (T) this;
	}

	/**
	 * Indicates whether the security interceptors are to be installed (on) or
	 * not (off).
	 */
	@ModelNodeBinding(detypedName = "security")
	public Security security() {
		return this.security;
	}

	/**
	 * Indicates whether the security interceptors are to be installed (on) or
	 * not (off).
	 */
	@SuppressWarnings("unchecked")
	public T security(Security value) {
		Object oldValue = this.security;
		this.security = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("security", oldValue, value);
		return (T) this;
	}

	@Deprecated
	@SuppressWarnings("unchecked")
	public T transactions(final String transactions) {
		if (transactions == null) {
			this.transactions = null;
		} else {
			boolean found = false;
			for (Transactions e : Transactions.values()) {
				if (e.toString().equals(transactions)) {
					transactions(e);
					found = true;
					break;
				}
			}
			if (!found)
				throw new RuntimeException(String.format(
						"Value '%s' not valid. Valid values are: %s",
						transactions, Arrays.asList(Transactions.values())));
		}
		return (T) this;
	}

	/**
	 * Indicates whether the transactions interceptors are to be installed (on
	 * or spec) or not (off). The value 'on' enabled JTS while 'spec' enables a
	 * spec compliant mode (non JTS) that rejects incoming transaction contexts.
	 */
	@ModelNodeBinding(detypedName = "transactions")
	public Transactions transactions() {
		return this.transactions;
	}

	/**
	 * Indicates whether the transactions interceptors are to be installed (on
	 * or spec) or not (off). The value 'on' enabled JTS while 'spec' enables a
	 * spec compliant mode (non JTS) that rejects incoming transaction contexts.
	 */
	@SuppressWarnings("unchecked")
	public T transactions(Transactions value) {
		Object oldValue = this.transactions;
		this.transactions = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("transactions", oldValue, value);
		return (T) this;
	}

	/**
	 * Indicates whether the root context should be exported as
	 * corbaloc::address:port/NameService (on) or not (off).
	 */
	@ModelNodeBinding(detypedName = "export-corbaloc")
	public Boolean exportCorbaloc() {
		return this.exportCorbaloc;
	}

	/**
	 * Indicates whether the root context should be exported as
	 * corbaloc::address:port/NameService (on) or not (off).
	 */
	@SuppressWarnings("unchecked")
	public T exportCorbaloc(Boolean value) {
		Object oldValue = this.exportCorbaloc;
		this.exportCorbaloc = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("exportCorbaloc", oldValue, value);
		return (T) this;
	}

	/**
	 * The naming service root context.
	 */
	@ModelNodeBinding(detypedName = "root-context")
	public String rootContext() {
		return this.rootContext;
	}

	/**
	 * The naming service root context.
	 */
	@SuppressWarnings("unchecked")
	public T rootContext(String value) {
		Object oldValue = this.rootContext;
		this.rootContext = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("rootContext", oldValue, value);
		return (T) this;
	}

	/**
	 * The GIOP version to be used.
	 */
	@ModelNodeBinding(detypedName = "giop-version")
	public String giopVersion() {
		return this.giopVersion;
	}

	/**
	 * The GIOP version to be used.
	 */
	@SuppressWarnings("unchecked")
	public T giopVersion(String value) {
		Object oldValue = this.giopVersion;
		this.giopVersion = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("giopVersion", oldValue, value);
		return (T) this;
	}

	/**
	 * Persistent id of the server. Persistent object references are valid
	 * across many activactions of the server and they identify it using this
	 * property. As a result of that: many activations of the same server should
	 * have this property set to the same value, different server instances
	 * running on the same host should have different server ids.
	 */
	@ModelNodeBinding(detypedName = "persistent-server-id")
	public String persistentServerId() {
		return this.persistentServerId;
	}

	/**
	 * Persistent id of the server. Persistent object references are valid
	 * across many activactions of the server and they identify it using this
	 * property. As a result of that: many activations of the same server should
	 * have this property set to the same value, different server instances
	 * running on the same host should have different server ids.
	 */
	@SuppressWarnings("unchecked")
	public T persistentServerId(String value) {
		Object oldValue = this.persistentServerId;
		this.persistentServerId = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("persistentServerId", oldValue, value);
		return (T) this;
	}

	/**
	 * The name of the socket binding configuration that specifies the ORB port.
	 */
	@ModelNodeBinding(detypedName = "socket-binding")
	public String socketBinding() {
		return this.socketBinding;
	}

	/**
	 * The name of the socket binding configuration that specifies the ORB port.
	 */
	@SuppressWarnings("unchecked")
	public T socketBinding(String value) {
		Object oldValue = this.socketBinding;
		this.socketBinding = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("socketBinding", oldValue, value);
		return (T) this;
	}

	/**
	 * The name of the socket binding configuration that specifies the ORB SSL
	 * port.
	 */
	@ModelNodeBinding(detypedName = "ssl-socket-binding")
	public String sslSocketBinding() {
		return this.sslSocketBinding;
	}

	/**
	 * The name of the socket binding configuration that specifies the ORB SSL
	 * port.
	 */
	@SuppressWarnings("unchecked")
	public T sslSocketBinding(String value) {
		Object oldValue = this.sslSocketBinding;
		this.sslSocketBinding = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("sslSocketBinding", oldValue, value);
		return (T) this;
	}

	@Deprecated
	@SuppressWarnings("unchecked")
	public T callerPropagation(final String callerPropagation) {
		if (callerPropagation == null) {
			this.callerPropagation = null;
		} else {
			boolean found = false;
			for (CallerPropagation e : CallerPropagation.values()) {
				if (e.toString().equals(callerPropagation)) {
					callerPropagation(e);
					found = true;
					break;
				}
			}
			if (!found)
				throw new RuntimeException(String.format(
						"Value '%s' not valid. Valid values are: %s",
						callerPropagation,
						Arrays.asList(CallerPropagation.values())));
		}
		return (T) this;
	}

	/**
	 * Indicates whether the caller identity should be propagated in the SAS
	 * context or not. Valid values are 'none' and 'supported'.
	 */
	@ModelNodeBinding(detypedName = "caller-propagation")
	public CallerPropagation callerPropagation() {
		return this.callerPropagation;
	}

	/**
	 * Indicates whether the caller identity should be propagated in the SAS
	 * context or not. Valid values are 'none' and 'supported'.
	 */
	@SuppressWarnings("unchecked")
	public T callerPropagation(CallerPropagation value) {
		Object oldValue = this.callerPropagation;
		this.callerPropagation = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("callerPropagation", oldValue, value);
		return (T) this;
	}

	/**
	 * Indicates whether SSL components should be added by an IOR interceptor
	 * (on) or not (off).
	 */
	@ModelNodeBinding(detypedName = "add-component-via-interceptor")
	public Boolean addComponentViaInterceptor() {
		return this.addComponentViaInterceptor;
	}

	/**
	 * Indicates whether SSL components should be added by an IOR interceptor
	 * (on) or not (off).
	 */
	@SuppressWarnings("unchecked")
	public T addComponentViaInterceptor(Boolean value) {
		Object oldValue = this.addComponentViaInterceptor;
		this.addComponentViaInterceptor = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("addComponentViaInterceptor", oldValue,
					value);
		return (T) this;
	}

	@Deprecated
	@SuppressWarnings("unchecked")
	public T clientRequires(final String clientRequires) {
		if (clientRequires == null) {
			this.clientRequires = null;
		} else {
			boolean found = false;
			for (ClientRequires e : ClientRequires.values()) {
				if (e.toString().equals(clientRequires)) {
					clientRequires(e);
					found = true;
					break;
				}
			}
			if (!found)
				throw new RuntimeException(String.format(
						"Value '%s' not valid. Valid values are: %s",
						clientRequires, Arrays.asList(ClientRequires.values())));
		}
		return (T) this;
	}

	/**
	 * Value that indicates the client SSL required parameters (None,
	 * ServerAuth, ClientAuth, MutualAuth).
	 */
	@ModelNodeBinding(detypedName = "client-requires")
	public ClientRequires clientRequires() {
		return this.clientRequires;
	}

	/**
	 * Value that indicates the client SSL required parameters (None,
	 * ServerAuth, ClientAuth, MutualAuth).
	 */
	@SuppressWarnings("unchecked")
	public T clientRequires(ClientRequires value) {
		Object oldValue = this.clientRequires;
		this.clientRequires = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("clientRequires", oldValue, value);
		return (T) this;
	}

	@Deprecated
	@SuppressWarnings("unchecked")
	public T clientSupports(final String clientSupports) {
		if (clientSupports == null) {
			this.clientSupports = null;
		} else {
			boolean found = false;
			for (ClientSupports e : ClientSupports.values()) {
				if (e.toString().equals(clientSupports)) {
					clientSupports(e);
					found = true;
					break;
				}
			}
			if (!found)
				throw new RuntimeException(String.format(
						"Value '%s' not valid. Valid values are: %s",
						clientSupports, Arrays.asList(ClientSupports.values())));
		}
		return (T) this;
	}

	/**
	 * Value that indicates the client SSL supported parameters (None,
	 * ServerAuth, ClientAuth, MutualAuth).
	 */
	@ModelNodeBinding(detypedName = "client-supports")
	public ClientSupports clientSupports() {
		return this.clientSupports;
	}

	/**
	 * Value that indicates the client SSL supported parameters (None,
	 * ServerAuth, ClientAuth, MutualAuth).
	 */
	@SuppressWarnings("unchecked")
	public T clientSupports(ClientSupports value) {
		Object oldValue = this.clientSupports;
		this.clientSupports = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("clientSupports", oldValue, value);
		return (T) this;
	}

	/**
	 * The name of the security domain that holds the key and trust stores that
	 * will be used to establish SSL connections.
	 */
	@ModelNodeBinding(detypedName = "security-domain")
	public String securityDomain() {
		return this.securityDomain;
	}

	/**
	 * The name of the security domain that holds the key and trust stores that
	 * will be used to establish SSL connections.
	 */
	@SuppressWarnings("unchecked")
	public T securityDomain(String value) {
		Object oldValue = this.securityDomain;
		this.securityDomain = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("securityDomain", oldValue, value);
		return (T) this;
	}

	@Deprecated
	@SuppressWarnings("unchecked")
	public T serverRequires(final String serverRequires) {
		if (serverRequires == null) {
			this.serverRequires = null;
		} else {
			boolean found = false;
			for (ServerRequires e : ServerRequires.values()) {
				if (e.toString().equals(serverRequires)) {
					serverRequires(e);
					found = true;
					break;
				}
			}
			if (!found)
				throw new RuntimeException(String.format(
						"Value '%s' not valid. Valid values are: %s",
						serverRequires, Arrays.asList(ServerRequires.values())));
		}
		return (T) this;
	}

	/**
	 * Value that indicates the server SSL required parameters (None,
	 * ServerAuth, ClientAuth, MutualAuth).
	 */
	@ModelNodeBinding(detypedName = "server-requires")
	public ServerRequires serverRequires() {
		return this.serverRequires;
	}

	/**
	 * Value that indicates the server SSL required parameters (None,
	 * ServerAuth, ClientAuth, MutualAuth).
	 */
	@SuppressWarnings("unchecked")
	public T serverRequires(ServerRequires value) {
		Object oldValue = this.serverRequires;
		this.serverRequires = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("serverRequires", oldValue, value);
		return (T) this;
	}

	@Deprecated
	@SuppressWarnings("unchecked")
	public T serverSupports(final String serverSupports) {
		if (serverSupports == null) {
			this.serverSupports = null;
		} else {
			boolean found = false;
			for (ServerSupports e : ServerSupports.values()) {
				if (e.toString().equals(serverSupports)) {
					serverSupports(e);
					found = true;
					break;
				}
			}
			if (!found)
				throw new RuntimeException(String.format(
						"Value '%s' not valid. Valid values are: %s",
						serverSupports, Arrays.asList(ServerSupports.values())));
		}
		return (T) this;
	}

	/**
	 * Value that indicates the server SSL supported parameters (None,
	 * ServerAuth, ClientAuth, MutualAuth).
	 */
	@ModelNodeBinding(detypedName = "server-supports")
	public ServerSupports serverSupports() {
		return this.serverSupports;
	}

	/**
	 * Value that indicates the server SSL supported parameters (None,
	 * ServerAuth, ClientAuth, MutualAuth).
	 */
	@SuppressWarnings("unchecked")
	public T serverSupports(ServerSupports value) {
		Object oldValue = this.serverSupports;
		this.serverSupports = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("serverSupports", oldValue, value);
		return (T) this;
	}

	/**
	 * Indicates whether SSL is to be supported (on) or not (off).
	 */
	@ModelNodeBinding(detypedName = "support-ssl")
	public Boolean supportSsl() {
		return this.supportSsl;
	}

	/**
	 * Indicates whether SSL is to be supported (on) or not (off).
	 */
	@SuppressWarnings("unchecked")
	public T supportSsl(Boolean value) {
		Object oldValue = this.supportSsl;
		this.supportSsl = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("supportSsl", oldValue, value);
		return (T) this;
	}

	/**
	 * TCP connection cache parameter. Each time the number of connections
	 * exceeds this value ORB tries to reclaim connections. Number of reclaimed
	 * connections is specified by tcp-number-to-reclaim property. If this
	 * property is not set then it is configured as OpenJDK ORB default.
	 */
	@ModelNodeBinding(detypedName = "high-water-mark")
	public Integer highWaterMark() {
		return this.highWaterMark;
	}

	/**
	 * TCP connection cache parameter. Each time the number of connections
	 * exceeds this value ORB tries to reclaim connections. Number of reclaimed
	 * connections is specified by tcp-number-to-reclaim property. If this
	 * property is not set then it is configured as OpenJDK ORB default.
	 */
	@SuppressWarnings("unchecked")
	public T highWaterMark(Integer value) {
		Object oldValue = this.highWaterMark;
		this.highWaterMark = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("highWaterMark", oldValue, value);
		return (T) this;
	}

	/**
	 * TCP connection cache parameter. Each time number of connections exceeds
	 * tcp-high-water-mark property then ORB tries to reclaim connections.
	 * Number of reclaimed connections is specified by this property. If it is
	 * not set then it is configured as OpenJDK ORB default.
	 */
	@ModelNodeBinding(detypedName = "number-to-reclaim")
	public Integer numberToReclaim() {
		return this.numberToReclaim;
	}

	/**
	 * TCP connection cache parameter. Each time number of connections exceeds
	 * tcp-high-water-mark property then ORB tries to reclaim connections.
	 * Number of reclaimed connections is specified by this property. If it is
	 * not set then it is configured as OpenJDK ORB default.
	 */
	@SuppressWarnings("unchecked")
	public T numberToReclaim(Integer value) {
		Object oldValue = this.numberToReclaim;
		this.numberToReclaim = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("numberToReclaim", oldValue, value);
		return (T) this;
	}

	@Deprecated
	@SuppressWarnings("unchecked")
	public T confidentiality(final String confidentiality) {
		if (confidentiality == null) {
			this.confidentiality = null;
		} else {
			boolean found = false;
			for (Confidentiality e : Confidentiality.values()) {
				if (e.toString().equals(confidentiality)) {
					confidentiality(e);
					found = true;
					break;
				}
			}
			if (!found)
				throw new RuntimeException(String.format(
						"Value '%s' not valid. Valid values are: %s",
						confidentiality,
						Arrays.asList(Confidentiality.values())));
		}
		return (T) this;
	}

	/**
	 * Indicates whether the transport must require confidentiality protection
	 * or not. Valid values are 'none', 'supported' and 'required'.
	 */
	@ModelNodeBinding(detypedName = "confidentiality")
	public Confidentiality confidentiality() {
		return this.confidentiality;
	}

	/**
	 * Indicates whether the transport must require confidentiality protection
	 * or not. Valid values are 'none', 'supported' and 'required'.
	 */
	@SuppressWarnings("unchecked")
	public T confidentiality(Confidentiality value) {
		Object oldValue = this.confidentiality;
		this.confidentiality = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("confidentiality", oldValue, value);
		return (T) this;
	}

	@Deprecated
	@SuppressWarnings("unchecked")
	public T detectMisordering(final String detectMisordering) {
		if (detectMisordering == null) {
			this.detectMisordering = null;
		} else {
			boolean found = false;
			for (DetectMisordering e : DetectMisordering.values()) {
				if (e.toString().equals(detectMisordering)) {
					detectMisordering(e);
					found = true;
					break;
				}
			}
			if (!found)
				throw new RuntimeException(String.format(
						"Value '%s' not valid. Valid values are: %s",
						detectMisordering,
						Arrays.asList(DetectMisordering.values())));
		}
		return (T) this;
	}

	/**
	 * Indicates whether the transport must require misordering detection or
	 * not. Valid values are 'none', 'supported' and 'required'.
	 */
	@ModelNodeBinding(detypedName = "detect-misordering")
	public DetectMisordering detectMisordering() {
		return this.detectMisordering;
	}

	/**
	 * Indicates whether the transport must require misordering detection or
	 * not. Valid values are 'none', 'supported' and 'required'.
	 */
	@SuppressWarnings("unchecked")
	public T detectMisordering(DetectMisordering value) {
		Object oldValue = this.detectMisordering;
		this.detectMisordering = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("detectMisordering", oldValue, value);
		return (T) this;
	}

	@Deprecated
	@SuppressWarnings("unchecked")
	public T detectReplay(final String detectReplay) {
		if (detectReplay == null) {
			this.detectReplay = null;
		} else {
			boolean found = false;
			for (DetectReplay e : DetectReplay.values()) {
				if (e.toString().equals(detectReplay)) {
					detectReplay(e);
					found = true;
					break;
				}
			}
			if (!found)
				throw new RuntimeException(String.format(
						"Value '%s' not valid. Valid values are: %s",
						detectReplay, Arrays.asList(DetectReplay.values())));
		}
		return (T) this;
	}

	/**
	 * Indicates whether the transport must require replay detection or not.
	 * Valid values are 'none', 'supported' and 'required'.
	 */
	@ModelNodeBinding(detypedName = "detect-replay")
	public DetectReplay detectReplay() {
		return this.detectReplay;
	}

	/**
	 * Indicates whether the transport must require replay detection or not.
	 * Valid values are 'none', 'supported' and 'required'.
	 */
	@SuppressWarnings("unchecked")
	public T detectReplay(DetectReplay value) {
		Object oldValue = this.detectReplay;
		this.detectReplay = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("detectReplay", oldValue, value);
		return (T) this;
	}

	@Deprecated
	@SuppressWarnings("unchecked")
	public T integrity(final String integrity) {
		if (integrity == null) {
			this.integrity = null;
		} else {
			boolean found = false;
			for (Integrity e : Integrity.values()) {
				if (e.toString().equals(integrity)) {
					integrity(e);
					found = true;
					break;
				}
			}
			if (!found)
				throw new RuntimeException(String.format(
						"Value '%s' not valid. Valid values are: %s",
						integrity, Arrays.asList(Integrity.values())));
		}
		return (T) this;
	}

	/**
	 * Indicates whether the transport must require integrity protection or not.
	 * Valid values are 'none', 'supported' and 'required'.
	 */
	@ModelNodeBinding(detypedName = "integrity")
	public Integrity integrity() {
		return this.integrity;
	}

	/**
	 * Indicates whether the transport must require integrity protection or not.
	 * Valid values are 'none', 'supported' and 'required'.
	 */
	@SuppressWarnings("unchecked")
	public T integrity(Integrity value) {
		Object oldValue = this.integrity;
		this.integrity = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("integrity", oldValue, value);
		return (T) this;
	}

	@Deprecated
	@SuppressWarnings("unchecked")
	public T trustInClient(final String trustInClient) {
		if (trustInClient == null) {
			this.trustInClient = null;
		} else {
			boolean found = false;
			for (TrustInClient e : TrustInClient.values()) {
				if (e.toString().equals(trustInClient)) {
					trustInClient(e);
					found = true;
					break;
				}
			}
			if (!found)
				throw new RuntimeException(String.format(
						"Value '%s' not valid. Valid values are: %s",
						trustInClient, Arrays.asList(TrustInClient.values())));
		}
		return (T) this;
	}

	/**
	 * Indicates if the transport must require trust in client to be
	 * established. Valid values are 'none', 'supported' and 'required'.
	 */
	@ModelNodeBinding(detypedName = "trust-in-client")
	public TrustInClient trustInClient() {
		return this.trustInClient;
	}

	/**
	 * Indicates if the transport must require trust in client to be
	 * established. Valid values are 'none', 'supported' and 'required'.
	 */
	@SuppressWarnings("unchecked")
	public T trustInClient(TrustInClient value) {
		Object oldValue = this.trustInClient;
		this.trustInClient = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("trustInClient", oldValue, value);
		return (T) this;
	}

	@Deprecated
	@SuppressWarnings("unchecked")
	public T trustInTarget(final String trustInTarget) {
		if (trustInTarget == null) {
			this.trustInTarget = null;
		} else {
			boolean found = false;
			for (TrustInTarget e : TrustInTarget.values()) {
				if (e.toString().equals(trustInTarget)) {
					trustInTarget(e);
					found = true;
					break;
				}
			}
			if (!found)
				throw new RuntimeException(String.format(
						"Value '%s' not valid. Valid values are: %s",
						trustInTarget, Arrays.asList(TrustInTarget.values())));
		}
		return (T) this;
	}

	/**
	 * Indicates if the transport must require trust in target to be
	 * established. Valid values are 'none' and 'supported'.
	 */
	@ModelNodeBinding(detypedName = "trust-in-target")
	public TrustInTarget trustInTarget() {
		return this.trustInTarget;
	}

	/**
	 * Indicates if the transport must require trust in target to be
	 * established. Valid values are 'none' and 'supported'.
	 */
	@SuppressWarnings("unchecked")
	public T trustInTarget(TrustInTarget value) {
		Object oldValue = this.trustInTarget;
		this.trustInTarget = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("trustInTarget", oldValue, value);
		return (T) this;
	}

	public static enum AuthMethod {
		NONE("none"), USERNAME_PASSWORD("username_password");
		private final String allowedValue;

		/**
		 * Returns the allowed value for the management model.
		 * 
		 * @return the allowed model value
		 */
		public String getAllowedValue() {
			return allowedValue;
		}

		AuthMethod(String allowedValue) {
			this.allowedValue = allowedValue;
		}

		@Override
		public String toString() {
			return allowedValue;
		}
	}

	public static enum Security {
		IDENTITY("identity"), CLIENT("client"), NONE("none");
		private final String allowedValue;

		/**
		 * Returns the allowed value for the management model.
		 * 
		 * @return the allowed model value
		 */
		public String getAllowedValue() {
			return allowedValue;
		}

		Security(String allowedValue) {
			this.allowedValue = allowedValue;
		}

		@Override
		public String toString() {
			return allowedValue;
		}
	}

	public static enum Transactions {
		FULL("full"), NONE("none"), SPEC("spec");
		private final String allowedValue;

		/**
		 * Returns the allowed value for the management model.
		 * 
		 * @return the allowed model value
		 */
		public String getAllowedValue() {
			return allowedValue;
		}

		Transactions(String allowedValue) {
			this.allowedValue = allowedValue;
		}

		@Override
		public String toString() {
			return allowedValue;
		}
	}

	public static enum CallerPropagation {
		NONE("none"), SUPPORTED("supported");
		private final String allowedValue;

		/**
		 * Returns the allowed value for the management model.
		 * 
		 * @return the allowed model value
		 */
		public String getAllowedValue() {
			return allowedValue;
		}

		CallerPropagation(String allowedValue) {
			this.allowedValue = allowedValue;
		}

		@Override
		public String toString() {
			return allowedValue;
		}
	}

	public static enum ClientRequires {
		NONE("None"), SERVERAUTH("ServerAuth"), CLIENTAUTH("ClientAuth"), MUTUALAUTH(
				"MutualAuth");
		private final String allowedValue;

		/**
		 * Returns the allowed value for the management model.
		 * 
		 * @return the allowed model value
		 */
		public String getAllowedValue() {
			return allowedValue;
		}

		ClientRequires(String allowedValue) {
			this.allowedValue = allowedValue;
		}

		@Override
		public String toString() {
			return allowedValue;
		}
	}

	public static enum ClientSupports {
		NONE("None"), SERVERAUTH("ServerAuth"), CLIENTAUTH("ClientAuth"), MUTUALAUTH(
				"MutualAuth");
		private final String allowedValue;

		/**
		 * Returns the allowed value for the management model.
		 * 
		 * @return the allowed model value
		 */
		public String getAllowedValue() {
			return allowedValue;
		}

		ClientSupports(String allowedValue) {
			this.allowedValue = allowedValue;
		}

		@Override
		public String toString() {
			return allowedValue;
		}
	}

	public static enum ServerRequires {
		NONE("None"), SERVERAUTH("ServerAuth"), CLIENTAUTH("ClientAuth"), MUTUALAUTH(
				"MutualAuth");
		private final String allowedValue;

		/**
		 * Returns the allowed value for the management model.
		 * 
		 * @return the allowed model value
		 */
		public String getAllowedValue() {
			return allowedValue;
		}

		ServerRequires(String allowedValue) {
			this.allowedValue = allowedValue;
		}

		@Override
		public String toString() {
			return allowedValue;
		}
	}

	public static enum ServerSupports {
		NONE("None"), SERVERAUTH("ServerAuth"), CLIENTAUTH("ClientAuth"), MUTUALAUTH(
				"MutualAuth");
		private final String allowedValue;

		/**
		 * Returns the allowed value for the management model.
		 * 
		 * @return the allowed model value
		 */
		public String getAllowedValue() {
			return allowedValue;
		}

		ServerSupports(String allowedValue) {
			this.allowedValue = allowedValue;
		}

		@Override
		public String toString() {
			return allowedValue;
		}
	}

	public static enum Confidentiality {
		NONE("none"), SUPPORTED("supported"), REQUIRED("required");
		private final String allowedValue;

		/**
		 * Returns the allowed value for the management model.
		 * 
		 * @return the allowed model value
		 */
		public String getAllowedValue() {
			return allowedValue;
		}

		Confidentiality(String allowedValue) {
			this.allowedValue = allowedValue;
		}

		@Override
		public String toString() {
			return allowedValue;
		}
	}

	public static enum DetectMisordering {
		NONE("none"), SUPPORTED("supported"), REQUIRED("required");
		private final String allowedValue;

		/**
		 * Returns the allowed value for the management model.
		 * 
		 * @return the allowed model value
		 */
		public String getAllowedValue() {
			return allowedValue;
		}

		DetectMisordering(String allowedValue) {
			this.allowedValue = allowedValue;
		}

		@Override
		public String toString() {
			return allowedValue;
		}
	}

	public static enum DetectReplay {
		NONE("none"), SUPPORTED("supported"), REQUIRED("required");
		private final String allowedValue;

		/**
		 * Returns the allowed value for the management model.
		 * 
		 * @return the allowed model value
		 */
		public String getAllowedValue() {
			return allowedValue;
		}

		DetectReplay(String allowedValue) {
			this.allowedValue = allowedValue;
		}

		@Override
		public String toString() {
			return allowedValue;
		}
	}

	public static enum Integrity {
		NONE("none"), SUPPORTED("supported"), REQUIRED("required");
		private final String allowedValue;

		/**
		 * Returns the allowed value for the management model.
		 * 
		 * @return the allowed model value
		 */
		public String getAllowedValue() {
			return allowedValue;
		}

		Integrity(String allowedValue) {
			this.allowedValue = allowedValue;
		}

		@Override
		public String toString() {
			return allowedValue;
		}
	}

	public static enum TrustInClient {
		NONE("none"), SUPPORTED("supported"), REQUIRED("required");
		private final String allowedValue;

		/**
		 * Returns the allowed value for the management model.
		 * 
		 * @return the allowed model value
		 */
		public String getAllowedValue() {
			return allowedValue;
		}

		TrustInClient(String allowedValue) {
			this.allowedValue = allowedValue;
		}

		@Override
		public String toString() {
			return allowedValue;
		}
	}

	public static enum TrustInTarget {
		NONE("none"), SUPPORTED("supported");
		private final String allowedValue;

		/**
		 * Returns the allowed value for the management model.
		 * 
		 * @return the allowed model value
		 */
		public String getAllowedValue() {
			return allowedValue;
		}

		TrustInTarget(String allowedValue) {
			this.allowedValue = allowedValue;
		}

		@Override
		public String toString() {
			return allowedValue;
		}
	}
}