package org.wildfly.swarm.config.infinispan.cache_container;

import org.wildfly.swarm.config.runtime.Addresses;
import org.wildfly.swarm.config.runtime.ResourceType;
import org.wildfly.swarm.config.runtime.Implicit;
import java.beans.PropertyChangeSupport;
import java.beans.PropertyChangeListener;
import org.wildfly.swarm.config.runtime.ModelNodeBinding;
/**
 * Configures a cache store as write-behind instead of write-through.
 */
@Addresses({
		"/subsystem=infinispan/cache-container=*/invalidation-cache=*/store=string-jdbc/write=behind",
		"/subsystem=infinispan/cache-container=*/invalidation-cache=*/store=custom/write=behind",
		"/subsystem=infinispan/cache-container=*/invalidation-cache=*/store=file/write=behind",
		"/subsystem=infinispan/cache-container=*/invalidation-cache=*/store=binary-jdbc/write=behind",
		"/subsystem=infinispan/cache-container=*/invalidation-cache=*/store=mixed-jdbc/write=behind",
		"/subsystem=infinispan/cache-container=*/invalidation-cache=*/store=remote/write=behind",
		"/subsystem=infinispan/cache-container=*/local-cache=*/store=string-jdbc/write=behind",
		"/subsystem=infinispan/cache-container=*/local-cache=*/store=custom/write=behind",
		"/subsystem=infinispan/cache-container=*/local-cache=*/store=file/write=behind",
		"/subsystem=infinispan/cache-container=*/local-cache=*/store=binary-jdbc/write=behind",
		"/subsystem=infinispan/cache-container=*/local-cache=*/store=mixed-jdbc/write=behind",
		"/subsystem=infinispan/cache-container=*/local-cache=*/store=remote/write=behind",
		"/subsystem=infinispan/cache-container=*/replicated-cache=*/store=custom/write=behind",
		"/subsystem=infinispan/cache-container=*/replicated-cache=*/store=binary-jdbc/write=behind",
		"/subsystem=infinispan/cache-container=*/replicated-cache=*/store=string-jdbc/write=behind",
		"/subsystem=infinispan/cache-container=*/replicated-cache=*/store=file/write=behind",
		"/subsystem=infinispan/cache-container=*/replicated-cache=*/store=mixed-jdbc/write=behind",
		"/subsystem=infinispan/cache-container=*/replicated-cache=*/store=remote/write=behind",
		"/subsystem=infinispan/cache-container=*/distributed-cache=*/store=custom/write=behind",
		"/subsystem=infinispan/cache-container=*/distributed-cache=*/store=binary-jdbc/write=behind",
		"/subsystem=infinispan/cache-container=*/distributed-cache=*/store=string-jdbc/write=behind",
		"/subsystem=infinispan/cache-container=*/distributed-cache=*/store=file/write=behind",
		"/subsystem=infinispan/cache-container=*/distributed-cache=*/store=mixed-jdbc/write=behind",
		"/subsystem=infinispan/cache-container=*/distributed-cache=*/store=remote/write=behind"})
@ResourceType("write")
@Implicit
public class BehindWrite<T extends BehindWrite<T>> {

	private String key;
	private PropertyChangeSupport pcs;
	private Integer modificationQueueSize;
	private Integer threadPoolSize;

	public BehindWrite() {
		this.key = "behind";
		this.pcs = new PropertyChangeSupport(this);
	}

	public String getKey() {
		return this.key;
	}

	/**
	 * Adds a property change listener
	 */
	public void addPropertyChangeListener(PropertyChangeListener listener) {
		if (null == this.pcs)
			this.pcs = new PropertyChangeSupport(this);
		this.pcs.addPropertyChangeListener(listener);
	}

	/**
	 * Removes a property change listener
	 */
	public void removePropertyChangeListener(PropertyChangeListener listener) {
		if (this.pcs != null)
			this.pcs.removePropertyChangeListener(listener);
	}

	/**
	 * Maximum number of entries in the asynchronous queue. When the queue is
	 * full, the store becomes write-through until it can accept new entries.
	 */
	@ModelNodeBinding(detypedName = "modification-queue-size")
	public Integer modificationQueueSize() {
		return this.modificationQueueSize;
	}

	/**
	 * Maximum number of entries in the asynchronous queue. When the queue is
	 * full, the store becomes write-through until it can accept new entries.
	 */
	@SuppressWarnings("unchecked")
	public T modificationQueueSize(Integer value) {
		Object oldValue = this.modificationQueueSize;
		this.modificationQueueSize = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("modificationQueueSize", oldValue,
					value);
		return (T) this;
	}

	/**
	 * Size of the thread pool whose threads are responsible for applying the
	 * modifications to the cache store.
	 */
	@ModelNodeBinding(detypedName = "thread-pool-size")
	public Integer threadPoolSize() {
		return this.threadPoolSize;
	}

	/**
	 * Size of the thread pool whose threads are responsible for applying the
	 * modifications to the cache store.
	 */
	@SuppressWarnings("unchecked")
	public T threadPoolSize(Integer value) {
		Object oldValue = this.threadPoolSize;
		this.threadPoolSize = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("threadPoolSize", oldValue, value);
		return (T) this;
	}
}