package org.wildfly.swarm.config.jgroups.channel;

import org.wildfly.swarm.config.runtime.Address;
import org.wildfly.swarm.config.runtime.ResourceType;
import java.beans.PropertyChangeSupport;
import java.beans.PropertyChangeListener;
import org.wildfly.swarm.config.runtime.ModelNodeBinding;
import java.util.List;
import org.wildfly.swarm.config.runtime.Subresource;
import org.wildfly.swarm.config.jgroups.ProtocolConsumer;
import org.wildfly.swarm.config.jgroups.ProtocolSupplier;
import org.wildfly.swarm.config.jgroups.Protocol;
/**
 * A JGroups channel fork
 */
@Address("/subsystem=jgroups/channel=*/fork=*")
@ResourceType("fork")
public class Fork<T extends Fork<T>> {

	private String key;
	private PropertyChangeSupport pcs;
	private ForkResources subresources = new ForkResources();

	public Fork(String key) {
		this.key = key;
	}

	public String getKey() {
		return this.key;
	}

	/**
	 * Adds a property change listener
	 */
	public void addPropertyChangeListener(PropertyChangeListener listener) {
		if (null == this.pcs)
			this.pcs = new PropertyChangeSupport(this);
		this.pcs.addPropertyChangeListener(listener);
	}

	/**
	 * Removes a property change listener
	 */
	public void removePropertyChangeListener(PropertyChangeListener listener) {
		if (this.pcs != null)
			this.pcs.removePropertyChangeListener(listener);
	}

	public ForkResources subresources() {
		return this.subresources;
	}

	/**
	 * Add all Protocol objects to this subresource
	 * 
	 * @return this
	 * @param value
	 *            List of Protocol objects.
	 */
	@SuppressWarnings("unchecked")
	public T protocols(List<Protocol> value) {
		this.subresources.protocols = value;
		return (T) this;
	}

	/**
	 * Add the Protocol object to the list of subresources
	 * 
	 * @param value
	 *            The Protocol to add
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T protocol(Protocol value) {
		this.subresources.protocols.add(value);
		return (T) this;
	}

	/**
	 * Create and configure a Protocol object to the list of subresources
	 * 
	 * @param key
	 *            The key for the Protocol resource
	 * @param config
	 *            The ProtocolConsumer to use
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T protocol(String childKey, ProtocolConsumer consumer) {
		Protocol<? extends Protocol> child = new Protocol<>(childKey);
		if (consumer != null) {
			consumer.accept(child);
		}
		protocol(child);
		return (T) this;
	}

	/**
	 * Create and configure a Protocol object to the list of subresources
	 * 
	 * @param key
	 *            The key for the Protocol resource
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T protocol(String childKey) {
		protocol(childKey, null);
		return (T) this;
	}

	/**
	 * Install a supplied Protocol object to the list of subresources
	 */
	@SuppressWarnings("unchecked")
	public T protocol(ProtocolSupplier supplier) {
		protocol(supplier.get());
		return (T) this;
	}

	/**
	 * Child mutators for Fork
	 */
	public static class ForkResources {
		/**
		 * The configuration of a protocol within a protocol stack.
		 */
		private List<Protocol> protocols = new java.util.ArrayList<>();

		/**
		 * Get the list of Protocol resources
		 * 
		 * @return the list of resources
		 */
		@Subresource
		public List<Protocol> protocols() {
			return this.protocols;
		}

		public Protocol protocol(String key) {
			return this.protocols.stream().filter(e -> e.getKey().equals(key))
					.findFirst().orElse(null);
		}
	}
}