package org.wildfly.swarm.config.management.access;

import org.wildfly.swarm.config.runtime.Address;
import org.wildfly.swarm.config.runtime.ResourceType;
import org.wildfly.swarm.config.runtime.Implicit;
import java.beans.PropertyChangeSupport;
import java.beans.PropertyChangeListener;
import org.wildfly.swarm.config.runtime.ModelNodeBinding;
import java.util.List;
import org.wildfly.swarm.config.runtime.Subresource;
import org.wildfly.swarm.config.management.access.constraint.TypeConsumer;
import org.wildfly.swarm.config.management.access.constraint.TypeSupplier;
import org.wildfly.swarm.config.management.access.constraint.Type;
/**
 * Configuration of the application classification constraints.
 */
@Address("/core-service=management/access=authorization/constraint=application-classification")
@ResourceType("constraint")
@Implicit
public class ApplicationClassificationConstraint<T extends ApplicationClassificationConstraint<T>> {

	private String key;
	private PropertyChangeSupport pcs;
	private ApplicationClassificationConstraintResources subresources = new ApplicationClassificationConstraintResources();

	public ApplicationClassificationConstraint() {
		this.key = "application-classification";
		this.pcs = new PropertyChangeSupport(this);
	}

	public String getKey() {
		return this.key;
	}

	/**
	 * Adds a property change listener
	 */
	public void addPropertyChangeListener(PropertyChangeListener listener) {
		if (null == this.pcs)
			this.pcs = new PropertyChangeSupport(this);
		this.pcs.addPropertyChangeListener(listener);
	}

	/**
	 * Removes a property change listener
	 */
	public void removePropertyChangeListener(PropertyChangeListener listener) {
		if (this.pcs != null)
			this.pcs.removePropertyChangeListener(listener);
	}

	public ApplicationClassificationConstraintResources subresources() {
		return this.subresources;
	}

	/**
	 * Add all Type objects to this subresource
	 * 
	 * @return this
	 * @param value
	 *            List of Type objects.
	 */
	@SuppressWarnings("unchecked")
	public T types(List<Type> value) {
		this.subresources.types = value;
		return (T) this;
	}

	/**
	 * Add the Type object to the list of subresources
	 * 
	 * @param value
	 *            The Type to add
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T type(Type value) {
		this.subresources.types.add(value);
		return (T) this;
	}

	/**
	 * Create and configure a Type object to the list of subresources
	 * 
	 * @param key
	 *            The key for the Type resource
	 * @param config
	 *            The TypeConsumer to use
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T type(String childKey, TypeConsumer consumer) {
		Type<? extends Type> child = new Type<>(childKey);
		if (consumer != null) {
			consumer.accept(child);
		}
		type(child);
		return (T) this;
	}

	/**
	 * Create and configure a Type object to the list of subresources
	 * 
	 * @param key
	 *            The key for the Type resource
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T type(String childKey) {
		type(childKey, null);
		return (T) this;
	}

	/**
	 * Install a supplied Type object to the list of subresources
	 */
	@SuppressWarnings("unchecked")
	public T type(TypeSupplier supplier) {
		type(supplier.get());
		return (T) this;
	}

	/**
	 * Child mutators for ApplicationClassificationConstraint
	 */
	public static class ApplicationClassificationConstraintResources {
		/**
		 * The application classification constraints by type. Type is either
		 * 'core' or the name of a subsystem.
		 */
		private List<Type> types = new java.util.ArrayList<>();

		/**
		 * Get the list of Type resources
		 * 
		 * @return the list of resources
		 */
		@Subresource
		public List<Type> types() {
			return this.types;
		}

		public Type type(String key) {
			return this.types.stream().filter(e -> e.getKey().equals(key))
					.findFirst().orElse(null);
		}
	}
}