package org.wildfly.swarm.config.management.security_realm;

import org.wildfly.swarm.config.runtime.Address;
import org.wildfly.swarm.config.runtime.ResourceType;
import org.wildfly.swarm.config.runtime.Implicit;
import java.beans.PropertyChangeSupport;
import java.beans.PropertyChangeListener;
import org.wildfly.swarm.config.runtime.ModelNodeBinding;
import java.util.Arrays;
import java.util.List;
import org.wildfly.swarm.config.runtime.Subresource;
import org.wildfly.swarm.config.management.PropertyConsumer;
import org.wildfly.swarm.config.management.PropertySupplier;
import org.wildfly.swarm.config.management.Property;
/**
 * Configuration of a user store plug-in for use by the realm.
 */
@Address("/core-service=management/security-realm=*/authentication=plug-in")
@ResourceType("authentication")
@Implicit
public class PlugInAuthentication<T extends PlugInAuthentication<T>> {

	private String key;
	private PropertyChangeSupport pcs;
	private Mechanism mechanism;
	private String name;
	private PlugInAuthenticationResources subresources = new PlugInAuthenticationResources();

	public PlugInAuthentication() {
		this.key = "plug-in";
		this.pcs = new PropertyChangeSupport(this);
	}

	public String getKey() {
		return this.key;
	}

	/**
	 * Adds a property change listener
	 */
	public void addPropertyChangeListener(PropertyChangeListener listener) {
		if (null == this.pcs)
			this.pcs = new PropertyChangeSupport(this);
		this.pcs.addPropertyChangeListener(listener);
	}

	/**
	 * Removes a property change listener
	 */
	public void removePropertyChangeListener(PropertyChangeListener listener) {
		if (this.pcs != null)
			this.pcs.removePropertyChangeListener(listener);
	}

	@Deprecated
	@SuppressWarnings("unchecked")
	public T mechanism(final String mechanism) {
		if (mechanism == null) {
			this.mechanism = null;
		} else {
			boolean found = false;
			for (Mechanism e : Mechanism.values()) {
				if (e.toString().equals(mechanism)) {
					mechanism(e);
					found = true;
					break;
				}
			}
			if (!found)
				throw new RuntimeException(String.format(
						"Value '%s' not valid. Valid values are: %s",
						mechanism, Arrays.asList(Mechanism.values())));
		}
		return (T) this;
	}

	/**
	 * Allow the mechanism this plug-in is compatible with to be overridden from
	 * DIGEST.
	 */
	@ModelNodeBinding(detypedName = "mechanism")
	public Mechanism mechanism() {
		return this.mechanism;
	}

	/**
	 * Allow the mechanism this plug-in is compatible with to be overridden from
	 * DIGEST.
	 */
	@SuppressWarnings("unchecked")
	public T mechanism(Mechanism value) {
		Object oldValue = this.mechanism;
		this.mechanism = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("mechanism", oldValue, value);
		return (T) this;
	}

	/**
	 * The short name of the plug-in (as registered) to use.
	 */
	@ModelNodeBinding(detypedName = "name")
	public String name() {
		return this.name;
	}

	/**
	 * The short name of the plug-in (as registered) to use.
	 */
	@SuppressWarnings("unchecked")
	public T name(String value) {
		Object oldValue = this.name;
		this.name = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("name", oldValue, value);
		return (T) this;
	}

	public PlugInAuthenticationResources subresources() {
		return this.subresources;
	}

	/**
	 * Add all Property objects to this subresource
	 * 
	 * @return this
	 * @param value
	 *            List of Property objects.
	 */
	@SuppressWarnings("unchecked")
	public T properties(List<Property> value) {
		this.subresources.properties = value;
		return (T) this;
	}

	/**
	 * Add the Property object to the list of subresources
	 * 
	 * @param value
	 *            The Property to add
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T property(Property value) {
		this.subresources.properties.add(value);
		return (T) this;
	}

	/**
	 * Create and configure a Property object to the list of subresources
	 * 
	 * @param key
	 *            The key for the Property resource
	 * @param config
	 *            The PropertyConsumer to use
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T property(String childKey, PropertyConsumer consumer) {
		Property<? extends Property> child = new Property<>(childKey);
		if (consumer != null) {
			consumer.accept(child);
		}
		property(child);
		return (T) this;
	}

	/**
	 * Create and configure a Property object to the list of subresources
	 * 
	 * @param key
	 *            The key for the Property resource
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T property(String childKey) {
		property(childKey, null);
		return (T) this;
	}

	/**
	 * Install a supplied Property object to the list of subresources
	 */
	@SuppressWarnings("unchecked")
	public T property(PropertySupplier supplier) {
		property(supplier.get());
		return (T) this;
	}

	/**
	 * Child mutators for PlugInAuthentication
	 */
	public static class PlugInAuthenticationResources {
		/**
		 * A property within a security realm resource.
		 */
		private List<Property> properties = new java.util.ArrayList<>();

		/**
		 * Get the list of Property resources
		 * 
		 * @return the list of resources
		 */
		@Subresource
		public List<Property> properties() {
			return this.properties;
		}

		public Property property(String key) {
			return this.properties.stream().filter(e -> e.getKey().equals(key))
					.findFirst().orElse(null);
		}
	}

	public static enum Mechanism {
		DIGEST("DIGEST"), PLAIN("PLAIN");
		private final String allowedValue;

		/**
		 * Returns the allowed value for the management model.
		 * 
		 * @return the allowed model value
		 */
		public String getAllowedValue() {
			return allowedValue;
		}

		Mechanism(String allowedValue) {
			this.allowedValue = allowedValue;
		}

		@Override
		public String toString() {
			return allowedValue;
		}
	}
}