package org.wildfly.swarm.config.naming;

import org.wildfly.swarm.config.runtime.Address;
import org.wildfly.swarm.config.runtime.ResourceType;
import java.beans.PropertyChangeSupport;
import java.beans.PropertyChangeListener;
import org.wildfly.swarm.config.runtime.ModelNodeBinding;
import java.util.Arrays;
import java.util.Map;
/**
 * JNDI bindings for primitive types
 */
@Address("/subsystem=naming/binding=*")
@ResourceType("binding")
public class Binding<T extends Binding<T>> {

	private String key;
	private PropertyChangeSupport pcs;
	private BindingType bindingType;
	private Boolean cache;
	private String attributeClass;
	private Map environment;
	private String lookup;
	private String module;
	private String type;
	private String value;

	public Binding(String key) {
		this.key = key;
	}

	public String getKey() {
		return this.key;
	}

	/**
	 * Adds a property change listener
	 */
	public void addPropertyChangeListener(PropertyChangeListener listener) {
		if (null == this.pcs)
			this.pcs = new PropertyChangeSupport(this);
		this.pcs.addPropertyChangeListener(listener);
	}

	/**
	 * Removes a property change listener
	 */
	public void removePropertyChangeListener(PropertyChangeListener listener) {
		if (this.pcs != null)
			this.pcs.removePropertyChangeListener(listener);
	}

	@Deprecated
	@SuppressWarnings("unchecked")
	public T bindingType(final String bindingType) {
		if (bindingType == null) {
			this.bindingType = null;
		} else {
			boolean found = false;
			for (BindingType e : BindingType.values()) {
				if (e.toString().equals(bindingType)) {
					bindingType(e);
					found = true;
					break;
				}
			}
			if (!found)
				throw new RuntimeException(String.format(
						"Value '%s' not valid. Valid values are: %s",
						bindingType, Arrays.asList(BindingType.values())));
		}
		return (T) this;
	}

	/**
	 * The type of binding to create, may be simple, lookup, external-context or
	 * object-factory
	 */
	@ModelNodeBinding(detypedName = "binding-type")
	public BindingType bindingType() {
		return this.bindingType;
	}

	/**
	 * The type of binding to create, may be simple, lookup, external-context or
	 * object-factory
	 */
	@SuppressWarnings("unchecked")
	public T bindingType(BindingType value) {
		Object oldValue = this.bindingType;
		this.bindingType = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("bindingType", oldValue, value);
		return (T) this;
	}

	/**
	 * If the external context should be cached
	 */
	@ModelNodeBinding(detypedName = "cache")
	public Boolean cache() {
		return this.cache;
	}

	/**
	 * If the external context should be cached
	 */
	@SuppressWarnings("unchecked")
	public T cache(Boolean value) {
		Object oldValue = this.cache;
		this.cache = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("cache", oldValue, value);
		return (T) this;
	}

	/**
	 * The object factory class name for object factory bindings
	 */
	@ModelNodeBinding(detypedName = "class")
	public String attributeClass() {
		return this.attributeClass;
	}

	/**
	 * The object factory class name for object factory bindings
	 */
	@SuppressWarnings("unchecked")
	public T attributeClass(String value) {
		Object oldValue = this.attributeClass;
		this.attributeClass = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("attributeClass", oldValue, value);
		return (T) this;
	}

	/**
	 * The environment to use on object factory instance retrieval
	 */
	@ModelNodeBinding(detypedName = "environment")
	public Map environment() {
		return this.environment;
	}

	/**
	 * The environment to use on object factory instance retrieval
	 */
	@SuppressWarnings("unchecked")
	public T environment(Map value) {
		Object oldValue = this.environment;
		this.environment = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("environment", oldValue, value);
		return (T) this;
	}

	/**
	 * The environment to use on object factory instance retrieval
	 */
	public T environment(String key, Object value) {
		if (this.environment == null) {
			this.environment = new java.util.HashMap<>();
		}
		this.environment.put(key, value);
		return (T) this;
	}

	/**
	 * The entry to lookup in JNDI for lookup bindings
	 */
	@ModelNodeBinding(detypedName = "lookup")
	public String lookup() {
		return this.lookup;
	}

	/**
	 * The entry to lookup in JNDI for lookup bindings
	 */
	@SuppressWarnings("unchecked")
	public T lookup(String value) {
		Object oldValue = this.lookup;
		this.lookup = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("lookup", oldValue, value);
		return (T) this;
	}

	/**
	 * The module to load the object factory from for object factory bindings
	 */
	@ModelNodeBinding(detypedName = "module")
	public String module() {
		return this.module;
	}

	/**
	 * The module to load the object factory from for object factory bindings
	 */
	@SuppressWarnings("unchecked")
	public T module(String value) {
		Object oldValue = this.module;
		this.module = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("module", oldValue, value);
		return (T) this;
	}

	/**
	 * The type of the value to bind for simple bindings, this must be a
	 * primitive type
	 */
	@ModelNodeBinding(detypedName = "type")
	public String type() {
		return this.type;
	}

	/**
	 * The type of the value to bind for simple bindings, this must be a
	 * primitive type
	 */
	@SuppressWarnings("unchecked")
	public T type(String value) {
		Object oldValue = this.type;
		this.type = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("type", oldValue, value);
		return (T) this;
	}

	/**
	 * The value to bind for simple bindings
	 */
	@ModelNodeBinding(detypedName = "value")
	public String value() {
		return this.value;
	}

	/**
	 * The value to bind for simple bindings
	 */
	@SuppressWarnings("unchecked")
	public T value(String value) {
		Object oldValue = this.value;
		this.value = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("value", oldValue, value);
		return (T) this;
	}

	public static enum BindingType {
		SIMPLE("simple"), OBJECT_FACTORY("object-factory"), LOOKUP("lookup"), EXTERNAL_CONTEXT(
				"external-context");
		private final String allowedValue;

		/**
		 * Returns the allowed value for the management model.
		 * 
		 * @return the allowed model value
		 */
		public String getAllowedValue() {
			return allowedValue;
		}

		BindingType(String allowedValue) {
			this.allowedValue = allowedValue;
		}

		@Override
		public String toString() {
			return allowedValue;
		}
	}
}