package org.wildfly.swarm.config.undertow.configuration;

import org.wildfly.swarm.config.runtime.Address;
import org.wildfly.swarm.config.runtime.ResourceType;
import java.beans.PropertyChangeSupport;
import java.beans.PropertyChangeListener;
import org.wildfly.swarm.config.runtime.ModelNodeBinding;
import java.util.List;
import org.wildfly.swarm.config.runtime.Subresource;
import org.wildfly.swarm.config.undertow.configuration.reverse_proxy.HostConsumer;
import org.wildfly.swarm.config.undertow.configuration.reverse_proxy.HostSupplier;
import org.wildfly.swarm.config.undertow.configuration.reverse_proxy.Host;
/**
 * A reverse proxy handler
 */
@Address("/subsystem=undertow/configuration=handler/reverse-proxy=*")
@ResourceType("reverse-proxy")
public class ReverseProxy<T extends ReverseProxy<T>> {

	private String key;
	private PropertyChangeSupport pcs;
	private Integer cachedConnectionsPerThread;
	private Integer connectionIdleTimeout;
	private Integer connectionsPerThread;
	private Integer maxRequestTime;
	private Integer problemServerRetry;
	private Integer requestQueueSize;
	private String sessionCookieNames;
	private ReverseProxyResources subresources = new ReverseProxyResources();

	public ReverseProxy(String key) {
		this.key = key;
	}

	public String getKey() {
		return this.key;
	}

	/**
	 * Adds a property change listener
	 */
	public void addPropertyChangeListener(PropertyChangeListener listener) {
		if (null == this.pcs)
			this.pcs = new PropertyChangeSupport(this);
		this.pcs.addPropertyChangeListener(listener);
	}

	/**
	 * Removes a property change listener
	 */
	public void removePropertyChangeListener(PropertyChangeListener listener) {
		if (this.pcs != null)
			this.pcs.removePropertyChangeListener(listener);
	}

	/**
	 * The number of connections that will be kept alive indefinitely
	 */
	@ModelNodeBinding(detypedName = "cached-connections-per-thread")
	public Integer cachedConnectionsPerThread() {
		return this.cachedConnectionsPerThread;
	}

	/**
	 * The number of connections that will be kept alive indefinitely
	 */
	@SuppressWarnings("unchecked")
	public T cachedConnectionsPerThread(Integer value) {
		Object oldValue = this.cachedConnectionsPerThread;
		this.cachedConnectionsPerThread = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("cachedConnectionsPerThread", oldValue,
					value);
		return (T) this;
	}

	/**
	 * The amount of time a connection can be idle before it will be closed.
	 * Connections will not time out once the pool size is down to the
	 * configured minimum (as configured by cached-connections-per-thread)
	 */
	@ModelNodeBinding(detypedName = "connection-idle-timeout")
	public Integer connectionIdleTimeout() {
		return this.connectionIdleTimeout;
	}

	/**
	 * The amount of time a connection can be idle before it will be closed.
	 * Connections will not time out once the pool size is down to the
	 * configured minimum (as configured by cached-connections-per-thread)
	 */
	@SuppressWarnings("unchecked")
	public T connectionIdleTimeout(Integer value) {
		Object oldValue = this.connectionIdleTimeout;
		this.connectionIdleTimeout = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("connectionIdleTimeout", oldValue,
					value);
		return (T) this;
	}

	/**
	 * The number of connections that will be maintained to backend servers, per
	 * IO thread. Defaults to 10.
	 */
	@ModelNodeBinding(detypedName = "connections-per-thread")
	public Integer connectionsPerThread() {
		return this.connectionsPerThread;
	}

	/**
	 * The number of connections that will be maintained to backend servers, per
	 * IO thread. Defaults to 10.
	 */
	@SuppressWarnings("unchecked")
	public T connectionsPerThread(Integer value) {
		Object oldValue = this.connectionsPerThread;
		this.connectionsPerThread = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("connectionsPerThread", oldValue, value);
		return (T) this;
	}

	/**
	 * The maximum time that a proxy request can be active for, before being
	 * killed. Defaults to unlimited
	 */
	@ModelNodeBinding(detypedName = "max-request-time")
	public Integer maxRequestTime() {
		return this.maxRequestTime;
	}

	/**
	 * The maximum time that a proxy request can be active for, before being
	 * killed. Defaults to unlimited
	 */
	@SuppressWarnings("unchecked")
	public T maxRequestTime(Integer value) {
		Object oldValue = this.maxRequestTime;
		this.maxRequestTime = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("maxRequestTime", oldValue, value);
		return (T) this;
	}

	/**
	 * Time in seconds to wait before attempting to reconnect to a server that
	 * is down
	 */
	@ModelNodeBinding(detypedName = "problem-server-retry")
	public Integer problemServerRetry() {
		return this.problemServerRetry;
	}

	/**
	 * Time in seconds to wait before attempting to reconnect to a server that
	 * is down
	 */
	@SuppressWarnings("unchecked")
	public T problemServerRetry(Integer value) {
		Object oldValue = this.problemServerRetry;
		this.problemServerRetry = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("problemServerRetry", oldValue, value);
		return (T) this;
	}

	/**
	 * The number of requests that can be queued if the connection pool is full
	 * before requests are rejected with a 503
	 */
	@ModelNodeBinding(detypedName = "request-queue-size")
	public Integer requestQueueSize() {
		return this.requestQueueSize;
	}

	/**
	 * The number of requests that can be queued if the connection pool is full
	 * before requests are rejected with a 503
	 */
	@SuppressWarnings("unchecked")
	public T requestQueueSize(Integer value) {
		Object oldValue = this.requestQueueSize;
		this.requestQueueSize = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("requestQueueSize", oldValue, value);
		return (T) this;
	}

	/**
	 * Comma separated list of session cookie names. Generally this will just be
	 * JSESSIONID.
	 */
	@ModelNodeBinding(detypedName = "session-cookie-names")
	public String sessionCookieNames() {
		return this.sessionCookieNames;
	}

	/**
	 * Comma separated list of session cookie names. Generally this will just be
	 * JSESSIONID.
	 */
	@SuppressWarnings("unchecked")
	public T sessionCookieNames(String value) {
		Object oldValue = this.sessionCookieNames;
		this.sessionCookieNames = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("sessionCookieNames", oldValue, value);
		return (T) this;
	}

	public ReverseProxyResources subresources() {
		return this.subresources;
	}

	/**
	 * Add all Host objects to this subresource
	 * 
	 * @return this
	 * @param value
	 *            List of Host objects.
	 */
	@SuppressWarnings("unchecked")
	public T hosts(List<Host> value) {
		this.subresources.hosts = value;
		return (T) this;
	}

	/**
	 * Add the Host object to the list of subresources
	 * 
	 * @param value
	 *            The Host to add
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T host(Host value) {
		this.subresources.hosts.add(value);
		return (T) this;
	}

	/**
	 * Create and configure a Host object to the list of subresources
	 * 
	 * @param key
	 *            The key for the Host resource
	 * @param config
	 *            The HostConsumer to use
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T host(String childKey, HostConsumer consumer) {
		Host<? extends Host> child = new Host<>(childKey);
		if (consumer != null) {
			consumer.accept(child);
		}
		host(child);
		return (T) this;
	}

	/**
	 * Create and configure a Host object to the list of subresources
	 * 
	 * @param key
	 *            The key for the Host resource
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T host(String childKey) {
		host(childKey, null);
		return (T) this;
	}

	/**
	 * Install a supplied Host object to the list of subresources
	 */
	@SuppressWarnings("unchecked")
	public T host(HostSupplier supplier) {
		host(supplier.get());
		return (T) this;
	}

	/**
	 * Child mutators for ReverseProxy
	 */
	public static class ReverseProxyResources {
		/**
		 * A host that the reverse proxy will forward requests to
		 */
		private List<Host> hosts = new java.util.ArrayList<>();

		/**
		 * Get the list of Host resources
		 * 
		 * @return the list of resources
		 */
		@Subresource
		public List<Host> hosts() {
			return this.hosts;
		}

		public Host host(String key) {
			return this.hosts.stream().filter(e -> e.getKey().equals(key))
					.findFirst().orElse(null);
		}
	}
}