package org.wildfly.swarm.config.management.security_realm;

import org.wildfly.swarm.config.runtime.Address;
import org.wildfly.swarm.config.runtime.ResourceType;
import org.wildfly.swarm.config.runtime.Implicit;
import java.beans.PropertyChangeSupport;
import java.beans.PropertyChangeListener;
import org.wildfly.swarm.config.runtime.ModelNodeBinding;
/**
 * Configuration of a keystore to use to create a trust manager to verify
 * clients.
 */
@Address("/core-service=management/security-realm=*/authentication=truststore")
@ResourceType("authentication")
@Implicit
public class TruststoreAuthentication<T extends TruststoreAuthentication<T>> {

	private String key;
	private PropertyChangeSupport pcs;
	private String keystorePassword;
	private String keystorePath;
	private String keystoreProvider;
	private String keystoreRelativeTo;

	public TruststoreAuthentication() {
		this.key = "truststore";
		this.pcs = new PropertyChangeSupport(this);
	}

	public String getKey() {
		return this.key;
	}

	/**
	 * Adds a property change listener
	 */
	public void addPropertyChangeListener(PropertyChangeListener listener) {
		if (null == this.pcs)
			this.pcs = new PropertyChangeSupport(this);
		this.pcs.addPropertyChangeListener(listener);
	}

	/**
	 * Removes a property change listener
	 */
	public void removePropertyChangeListener(PropertyChangeListener listener) {
		if (this.pcs != null)
			this.pcs.removePropertyChangeListener(listener);
	}

	/**
	 * The password to open the keystore.
	 */
	@ModelNodeBinding(detypedName = "keystore-password")
	public String keystorePassword() {
		return this.keystorePassword;
	}

	/**
	 * The password to open the keystore.
	 */
	@SuppressWarnings("unchecked")
	public T keystorePassword(String value) {
		Object oldValue = this.keystorePassword;
		this.keystorePassword = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("keystorePassword", oldValue, value);
		return (T) this;
	}

	/**
	 * The path of the keystore, will be ignored if the keystore-provider is
	 * anything other than JKS.
	 */
	@ModelNodeBinding(detypedName = "keystore-path")
	public String keystorePath() {
		return this.keystorePath;
	}

	/**
	 * The path of the keystore, will be ignored if the keystore-provider is
	 * anything other than JKS.
	 */
	@SuppressWarnings("unchecked")
	public T keystorePath(String value) {
		Object oldValue = this.keystorePath;
		this.keystorePath = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("keystorePath", oldValue, value);
		return (T) this;
	}

	/**
	 * The provider for loading the keystore, defaults to JKS.
	 */
	@ModelNodeBinding(detypedName = "keystore-provider")
	public String keystoreProvider() {
		return this.keystoreProvider;
	}

	/**
	 * The provider for loading the keystore, defaults to JKS.
	 */
	@SuppressWarnings("unchecked")
	public T keystoreProvider(String value) {
		Object oldValue = this.keystoreProvider;
		this.keystoreProvider = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("keystoreProvider", oldValue, value);
		return (T) this;
	}

	/**
	 * The name of another previously named path, or of one of the standard
	 * paths provided by the system. If 'relative-to' is provided, the value of
	 * the 'path' attribute is treated as relative to the path specified by this
	 * attribute.
	 */
	@ModelNodeBinding(detypedName = "keystore-relative-to")
	public String keystoreRelativeTo() {
		return this.keystoreRelativeTo;
	}

	/**
	 * The name of another previously named path, or of one of the standard
	 * paths provided by the system. If 'relative-to' is provided, the value of
	 * the 'path' attribute is treated as relative to the path specified by this
	 * attribute.
	 */
	@SuppressWarnings("unchecked")
	public T keystoreRelativeTo(String value) {
		Object oldValue = this.keystoreRelativeTo;
		this.keystoreRelativeTo = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("keystoreRelativeTo", oldValue, value);
		return (T) this;
	}
}