package org.wildfly.swarm.config.undertow.configuration;

import org.wildfly.swarm.config.runtime.Address;
import org.wildfly.swarm.config.runtime.ResourceType;
import java.beans.PropertyChangeSupport;
import java.beans.PropertyChangeListener;
import org.wildfly.swarm.config.runtime.ModelNodeBinding;
import java.util.List;
import org.wildfly.swarm.config.runtime.Subresource;
import org.wildfly.swarm.config.undertow.configuration.mod_cluster.BalancerConsumer;
import org.wildfly.swarm.config.undertow.configuration.mod_cluster.BalancerSupplier;
import org.wildfly.swarm.config.undertow.configuration.mod_cluster.Balancer;
/**
 * A mod-cluster front end load balancer
 */
@Address("/subsystem=undertow/configuration=filter/mod-cluster=*")
@ResourceType("mod-cluster")
public class ModCluster<T extends ModCluster<T>> {

	private String key;
	private PropertyChangeSupport pcs;
	private Integer advertiseFrequency;
	private String advertisePath;
	private String advertiseProtocol;
	private String advertiseSocketBinding;
	private Integer brokenNodeTimeout;
	private Integer cachedConnectionsPerThread;
	private Integer connectionIdleTimeout;
	private Integer connectionsPerThread;
	private Integer healthCheckInterval;
	private String managementAccessPredicate;
	private String managementSocketBinding;
	private Integer maxRequestTime;
	private Integer requestQueueSize;
	private String securityKey;
	private String securityRealm;
	private Boolean useAlias;
	private String worker;
	private ModClusterResources subresources = new ModClusterResources();

	public ModCluster(String key) {
		this.key = key;
	}

	public String getKey() {
		return this.key;
	}

	/**
	 * Adds a property change listener
	 */
	public void addPropertyChangeListener(PropertyChangeListener listener) {
		if (null == this.pcs)
			this.pcs = new PropertyChangeSupport(this);
		this.pcs.addPropertyChangeListener(listener);
	}

	/**
	 * Removes a property change listener
	 */
	public void removePropertyChangeListener(PropertyChangeListener listener) {
		if (this.pcs != null)
			this.pcs.removePropertyChangeListener(listener);
	}

	/**
	 * The frequency (in milliseconds) that mod-cluster advertises itself on the
	 * network
	 */
	@ModelNodeBinding(detypedName = "advertise-frequency")
	public Integer advertiseFrequency() {
		return this.advertiseFrequency;
	}

	/**
	 * The frequency (in milliseconds) that mod-cluster advertises itself on the
	 * network
	 */
	@SuppressWarnings("unchecked")
	public T advertiseFrequency(Integer value) {
		Object oldValue = this.advertiseFrequency;
		this.advertiseFrequency = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("advertiseFrequency", oldValue, value);
		return (T) this;
	}

	/**
	 * The path that mod-cluster is registered under, defaults to /
	 */
	@ModelNodeBinding(detypedName = "advertise-path")
	public String advertisePath() {
		return this.advertisePath;
	}

	/**
	 * The path that mod-cluster is registered under, defaults to /
	 */
	@SuppressWarnings("unchecked")
	public T advertisePath(String value) {
		Object oldValue = this.advertisePath;
		this.advertisePath = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("advertisePath", oldValue, value);
		return (T) this;
	}

	/**
	 * The protocol that is in use, defaults to HTTP
	 */
	@ModelNodeBinding(detypedName = "advertise-protocol")
	public String advertiseProtocol() {
		return this.advertiseProtocol;
	}

	/**
	 * The protocol that is in use, defaults to HTTP
	 */
	@SuppressWarnings("unchecked")
	public T advertiseProtocol(String value) {
		Object oldValue = this.advertiseProtocol;
		this.advertiseProtocol = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("advertiseProtocol", oldValue, value);
		return (T) this;
	}

	/**
	 * The multicast group that is used to advertise
	 */
	@ModelNodeBinding(detypedName = "advertise-socket-binding")
	public String advertiseSocketBinding() {
		return this.advertiseSocketBinding;
	}

	/**
	 * The multicast group that is used to advertise
	 */
	@SuppressWarnings("unchecked")
	public T advertiseSocketBinding(String value) {
		Object oldValue = this.advertiseSocketBinding;
		this.advertiseSocketBinding = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("advertiseSocketBinding", oldValue,
					value);
		return (T) this;
	}

	/**
	 * The amount of time that must elapse before a broken node is removed from
	 * the table
	 */
	@ModelNodeBinding(detypedName = "broken-node-timeout")
	public Integer brokenNodeTimeout() {
		return this.brokenNodeTimeout;
	}

	/**
	 * The amount of time that must elapse before a broken node is removed from
	 * the table
	 */
	@SuppressWarnings("unchecked")
	public T brokenNodeTimeout(Integer value) {
		Object oldValue = this.brokenNodeTimeout;
		this.brokenNodeTimeout = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("brokenNodeTimeout", oldValue, value);
		return (T) this;
	}

	/**
	 * The number of connections that will be kept alive indefinitely
	 */
	@ModelNodeBinding(detypedName = "cached-connections-per-thread")
	public Integer cachedConnectionsPerThread() {
		return this.cachedConnectionsPerThread;
	}

	/**
	 * The number of connections that will be kept alive indefinitely
	 */
	@SuppressWarnings("unchecked")
	public T cachedConnectionsPerThread(Integer value) {
		Object oldValue = this.cachedConnectionsPerThread;
		this.cachedConnectionsPerThread = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("cachedConnectionsPerThread", oldValue,
					value);
		return (T) this;
	}

	/**
	 * The amount of time a connection can be idle before it will be closed.
	 * Connections will not time out once the pool size is down to the
	 * configured minimum (as configured by cached-connections-per-thread)
	 */
	@ModelNodeBinding(detypedName = "connection-idle-timeout")
	public Integer connectionIdleTimeout() {
		return this.connectionIdleTimeout;
	}

	/**
	 * The amount of time a connection can be idle before it will be closed.
	 * Connections will not time out once the pool size is down to the
	 * configured minimum (as configured by cached-connections-per-thread)
	 */
	@SuppressWarnings("unchecked")
	public T connectionIdleTimeout(Integer value) {
		Object oldValue = this.connectionIdleTimeout;
		this.connectionIdleTimeout = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("connectionIdleTimeout", oldValue,
					value);
		return (T) this;
	}

	/**
	 * The number of connections that will be maintained to backend servers, per
	 * IO thread. Defaults to 10.
	 */
	@ModelNodeBinding(detypedName = "connections-per-thread")
	public Integer connectionsPerThread() {
		return this.connectionsPerThread;
	}

	/**
	 * The number of connections that will be maintained to backend servers, per
	 * IO thread. Defaults to 10.
	 */
	@SuppressWarnings("unchecked")
	public T connectionsPerThread(Integer value) {
		Object oldValue = this.connectionsPerThread;
		this.connectionsPerThread = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("connectionsPerThread", oldValue, value);
		return (T) this;
	}

	/**
	 * The frequency of health check pings to backend nodes
	 */
	@ModelNodeBinding(detypedName = "health-check-interval")
	public Integer healthCheckInterval() {
		return this.healthCheckInterval;
	}

	/**
	 * The frequency of health check pings to backend nodes
	 */
	@SuppressWarnings("unchecked")
	public T healthCheckInterval(Integer value) {
		Object oldValue = this.healthCheckInterval;
		this.healthCheckInterval = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("healthCheckInterval", oldValue, value);
		return (T) this;
	}

	/**
	 * A predicate that is applied to incoming requests to determine if they can
	 * perform mod cluster management commands. Provides additional security on
	 * top of what is provided by limiting management to requests that originate
	 * from the management-socket-binding
	 */
	@ModelNodeBinding(detypedName = "management-access-predicate")
	public String managementAccessPredicate() {
		return this.managementAccessPredicate;
	}

	/**
	 * A predicate that is applied to incoming requests to determine if they can
	 * perform mod cluster management commands. Provides additional security on
	 * top of what is provided by limiting management to requests that originate
	 * from the management-socket-binding
	 */
	@SuppressWarnings("unchecked")
	public T managementAccessPredicate(String value) {
		Object oldValue = this.managementAccessPredicate;
		this.managementAccessPredicate = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("managementAccessPredicate", oldValue,
					value);
		return (T) this;
	}

	/**
	 * The socket binding of the mod_cluster management port. When using
	 * mod_cluster two HTTP listeners should be defined, a public one to handle
	 * requests, and one bound to the internal network to handle mod cluster
	 * commands. This socket binding should correspond to the internal listener,
	 * and should not be publicly accessible
	 */
	@ModelNodeBinding(detypedName = "management-socket-binding")
	public String managementSocketBinding() {
		return this.managementSocketBinding;
	}

	/**
	 * The socket binding of the mod_cluster management port. When using
	 * mod_cluster two HTTP listeners should be defined, a public one to handle
	 * requests, and one bound to the internal network to handle mod cluster
	 * commands. This socket binding should correspond to the internal listener,
	 * and should not be publicly accessible
	 */
	@SuppressWarnings("unchecked")
	public T managementSocketBinding(String value) {
		Object oldValue = this.managementSocketBinding;
		this.managementSocketBinding = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("managementSocketBinding", oldValue,
					value);
		return (T) this;
	}

	/**
	 * The max amount of time that a request to a backend node can take before
	 * it is killed
	 */
	@ModelNodeBinding(detypedName = "max-request-time")
	public Integer maxRequestTime() {
		return this.maxRequestTime;
	}

	/**
	 * The max amount of time that a request to a backend node can take before
	 * it is killed
	 */
	@SuppressWarnings("unchecked")
	public T maxRequestTime(Integer value) {
		Object oldValue = this.maxRequestTime;
		this.maxRequestTime = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("maxRequestTime", oldValue, value);
		return (T) this;
	}

	/**
	 * The number of requests that can be queued if the connection pool is full
	 * before requests are rejected with a 503
	 */
	@ModelNodeBinding(detypedName = "request-queue-size")
	public Integer requestQueueSize() {
		return this.requestQueueSize;
	}

	/**
	 * The number of requests that can be queued if the connection pool is full
	 * before requests are rejected with a 503
	 */
	@SuppressWarnings("unchecked")
	public T requestQueueSize(Integer value) {
		Object oldValue = this.requestQueueSize;
		this.requestQueueSize = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("requestQueueSize", oldValue, value);
		return (T) this;
	}

	/**
	 * The security key that is used for the mod-cluster group. All members must
	 * use the same security key.
	 */
	@ModelNodeBinding(detypedName = "security-key")
	public String securityKey() {
		return this.securityKey;
	}

	/**
	 * The security key that is used for the mod-cluster group. All members must
	 * use the same security key.
	 */
	@SuppressWarnings("unchecked")
	public T securityKey(String value) {
		Object oldValue = this.securityKey;
		this.securityKey = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("securityKey", oldValue, value);
		return (T) this;
	}

	/**
	 * The security realm that provides the SSL configuration
	 */
	@ModelNodeBinding(detypedName = "security-realm")
	public String securityRealm() {
		return this.securityRealm;
	}

	/**
	 * The security realm that provides the SSL configuration
	 */
	@SuppressWarnings("unchecked")
	public T securityRealm(String value) {
		Object oldValue = this.securityRealm;
		this.securityRealm = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("securityRealm", oldValue, value);
		return (T) this;
	}

	/**
	 * If an alias check is performed
	 */
	@ModelNodeBinding(detypedName = "use-alias")
	public Boolean useAlias() {
		return this.useAlias;
	}

	/**
	 * If an alias check is performed
	 */
	@SuppressWarnings("unchecked")
	public T useAlias(Boolean value) {
		Object oldValue = this.useAlias;
		this.useAlias = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("useAlias", oldValue, value);
		return (T) this;
	}

	/**
	 * The XNIO worker that is used to send the advertise notifications
	 */
	@ModelNodeBinding(detypedName = "worker")
	public String worker() {
		return this.worker;
	}

	/**
	 * The XNIO worker that is used to send the advertise notifications
	 */
	@SuppressWarnings("unchecked")
	public T worker(String value) {
		Object oldValue = this.worker;
		this.worker = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("worker", oldValue, value);
		return (T) this;
	}

	public ModClusterResources subresources() {
		return this.subresources;
	}

	/**
	 * Add all Balancer objects to this subresource
	 * 
	 * @return this
	 * @param value
	 *            List of Balancer objects.
	 */
	@SuppressWarnings("unchecked")
	public T balancers(List<Balancer> value) {
		this.subresources.balancers = value;
		return (T) this;
	}

	/**
	 * Add the Balancer object to the list of subresources
	 * 
	 * @param value
	 *            The Balancer to add
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T balancer(Balancer value) {
		this.subresources.balancers.add(value);
		return (T) this;
	}

	/**
	 * Create and configure a Balancer object to the list of subresources
	 * 
	 * @param key
	 *            The key for the Balancer resource
	 * @param config
	 *            The BalancerConsumer to use
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T balancer(String childKey, BalancerConsumer consumer) {
		Balancer<? extends Balancer> child = new Balancer<>(childKey);
		if (consumer != null) {
			consumer.accept(child);
		}
		balancer(child);
		return (T) this;
	}

	/**
	 * Create and configure a Balancer object to the list of subresources
	 * 
	 * @param key
	 *            The key for the Balancer resource
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T balancer(String childKey) {
		balancer(childKey, null);
		return (T) this;
	}

	/**
	 * Install a supplied Balancer object to the list of subresources
	 */
	@SuppressWarnings("unchecked")
	public T balancer(BalancerSupplier supplier) {
		balancer(supplier.get());
		return (T) this;
	}

	/**
	 * Child mutators for ModCluster
	 */
	public static class ModClusterResources {
		/**
		 * Runtime representation of a mod_cluster balancer
		 */
		private List<Balancer> balancers = new java.util.ArrayList<>();

		/**
		 * Get the list of Balancer resources
		 * 
		 * @return the list of resources
		 */
		@Subresource
		public List<Balancer> balancers() {
			return this.balancers;
		}

		public Balancer balancer(String key) {
			return this.balancers.stream().filter(e -> e.getKey().equals(key))
					.findFirst().orElse(null);
		}
	}
}