package org.wildfly.swarm.config.datasources;

import org.wildfly.swarm.config.runtime.Address;
import org.wildfly.swarm.config.runtime.ResourceType;
import java.beans.PropertyChangeSupport;
import java.beans.PropertyChangeListener;
import org.wildfly.swarm.config.runtime.ModelNodeBinding;
/**
 * Service that make a JDBC driver available for use in the runtime
 */
@Address("/subsystem=datasources/jdbc-driver=*")
@ResourceType("jdbc-driver")
public class JDBCDriver<T extends JDBCDriver<T>> {

	private String key;
	private PropertyChangeSupport pcs;
	private String deploymentName;
	private String driverClassName;
	private String driverDatasourceClassName;
	private Integer driverMajorVersion;
	private Integer driverMinorVersion;
	private String driverModuleName;
	private String driverName;
	private String driverXaDatasourceClassName;
	private Boolean jdbcCompliant;
	private String moduleSlot;
	private String profile;
	private String xaDatasourceClass;

	public JDBCDriver(String key) {
		this.key = key;
	}

	public String getKey() {
		return this.key;
	}

	/**
	 * Adds a property change listener
	 */
	public void addPropertyChangeListener(PropertyChangeListener listener) {
		if (null == this.pcs)
			this.pcs = new PropertyChangeSupport(this);
		this.pcs.addPropertyChangeListener(listener);
	}

	/**
	 * Removes a property change listener
	 */
	public void removePropertyChangeListener(PropertyChangeListener listener) {
		if (this.pcs != null)
			this.pcs.removePropertyChangeListener(listener);
	}

	/**
	 * The name of the deployment unit from which the driver was loaded
	 */
	@ModelNodeBinding(detypedName = "deployment-name")
	public String deploymentName() {
		return this.deploymentName;
	}

	/**
	 * The name of the deployment unit from which the driver was loaded
	 */
	@SuppressWarnings("unchecked")
	public T deploymentName(String value) {
		Object oldValue = this.deploymentName;
		this.deploymentName = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("deploymentName", oldValue, value);
		return (T) this;
	}

	/**
	 * The fully qualified class name of the java.sql.Driver implementation
	 */
	@ModelNodeBinding(detypedName = "driver-class-name")
	public String driverClassName() {
		return this.driverClassName;
	}

	/**
	 * The fully qualified class name of the java.sql.Driver implementation
	 */
	@SuppressWarnings("unchecked")
	public T driverClassName(String value) {
		Object oldValue = this.driverClassName;
		this.driverClassName = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("driverClassName", oldValue, value);
		return (T) this;
	}

	/**
	 * The fully qualified class name of the javax.sql.DataSource implementation
	 */
	@ModelNodeBinding(detypedName = "driver-datasource-class-name")
	public String driverDatasourceClassName() {
		return this.driverDatasourceClassName;
	}

	/**
	 * The fully qualified class name of the javax.sql.DataSource implementation
	 */
	@SuppressWarnings("unchecked")
	public T driverDatasourceClassName(String value) {
		Object oldValue = this.driverDatasourceClassName;
		this.driverDatasourceClassName = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("driverDatasourceClassName", oldValue,
					value);
		return (T) this;
	}

	/**
	 * The driver's major version number
	 */
	@ModelNodeBinding(detypedName = "driver-major-version")
	public Integer driverMajorVersion() {
		return this.driverMajorVersion;
	}

	/**
	 * The driver's major version number
	 */
	@SuppressWarnings("unchecked")
	public T driverMajorVersion(Integer value) {
		Object oldValue = this.driverMajorVersion;
		this.driverMajorVersion = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("driverMajorVersion", oldValue, value);
		return (T) this;
	}

	/**
	 * The driver's minor version number
	 */
	@ModelNodeBinding(detypedName = "driver-minor-version")
	public Integer driverMinorVersion() {
		return this.driverMinorVersion;
	}

	/**
	 * The driver's minor version number
	 */
	@SuppressWarnings("unchecked")
	public T driverMinorVersion(Integer value) {
		Object oldValue = this.driverMinorVersion;
		this.driverMinorVersion = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("driverMinorVersion", oldValue, value);
		return (T) this;
	}

	/**
	 * The name of the module from which the driver was loaded, if it was loaded
	 * from the module path
	 */
	@ModelNodeBinding(detypedName = "driver-module-name")
	public String driverModuleName() {
		return this.driverModuleName;
	}

	/**
	 * The name of the module from which the driver was loaded, if it was loaded
	 * from the module path
	 */
	@SuppressWarnings("unchecked")
	public T driverModuleName(String value) {
		Object oldValue = this.driverModuleName;
		this.driverModuleName = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("driverModuleName", oldValue, value);
		return (T) this;
	}

	/**
	 * Defines the JDBC driver the datasource should use. It is a symbolic name
	 * matching the the name of installed driver. In case the driver is deployed
	 * as jar, the name is the name of deployment unit
	 */
	@ModelNodeBinding(detypedName = "driver-name")
	public String driverName() {
		return this.driverName;
	}

	/**
	 * Defines the JDBC driver the datasource should use. It is a symbolic name
	 * matching the the name of installed driver. In case the driver is deployed
	 * as jar, the name is the name of deployment unit
	 */
	@SuppressWarnings("unchecked")
	public T driverName(String value) {
		Object oldValue = this.driverName;
		this.driverName = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("driverName", oldValue, value);
		return (T) this;
	}

	/**
	 * The fully qualified class name of the javax.sql.XADataSource
	 * implementation
	 */
	@ModelNodeBinding(detypedName = "driver-xa-datasource-class-name")
	public String driverXaDatasourceClassName() {
		return this.driverXaDatasourceClassName;
	}

	/**
	 * The fully qualified class name of the javax.sql.XADataSource
	 * implementation
	 */
	@SuppressWarnings("unchecked")
	public T driverXaDatasourceClassName(String value) {
		Object oldValue = this.driverXaDatasourceClassName;
		this.driverXaDatasourceClassName = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("driverXaDatasourceClassName",
					oldValue, value);
		return (T) this;
	}

	/**
	 * Whether or not the driver is JDBC compliant
	 */
	@ModelNodeBinding(detypedName = "jdbc-compliant")
	public Boolean jdbcCompliant() {
		return this.jdbcCompliant;
	}

	/**
	 * Whether or not the driver is JDBC compliant
	 */
	@SuppressWarnings("unchecked")
	public T jdbcCompliant(Boolean value) {
		Object oldValue = this.jdbcCompliant;
		this.jdbcCompliant = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("jdbcCompliant", oldValue, value);
		return (T) this;
	}

	/**
	 * The slot of the module from which the driver was loaded, if it was loaded
	 * from the module path
	 */
	@ModelNodeBinding(detypedName = "module-slot")
	public String moduleSlot() {
		return this.moduleSlot;
	}

	/**
	 * The slot of the module from which the driver was loaded, if it was loaded
	 * from the module path
	 */
	@SuppressWarnings("unchecked")
	public T moduleSlot(String value) {
		Object oldValue = this.moduleSlot;
		this.moduleSlot = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("moduleSlot", oldValue, value);
		return (T) this;
	}

	/**
	 * Domain Profile in which driver is defined. Null in case of standalone
	 * server
	 */
	@ModelNodeBinding(detypedName = "profile")
	public String profile() {
		return this.profile;
	}

	/**
	 * Domain Profile in which driver is defined. Null in case of standalone
	 * server
	 */
	@SuppressWarnings("unchecked")
	public T profile(String value) {
		Object oldValue = this.profile;
		this.profile = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("profile", oldValue, value);
		return (T) this;
	}

	/**
	 * XA datasource class
	 */
	@ModelNodeBinding(detypedName = "xa-datasource-class")
	public String xaDatasourceClass() {
		return this.xaDatasourceClass;
	}

	/**
	 * XA datasource class
	 */
	@SuppressWarnings("unchecked")
	public T xaDatasourceClass(String value) {
		Object oldValue = this.xaDatasourceClass;
		this.xaDatasourceClass = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("xaDatasourceClass", oldValue, value);
		return (T) this;
	}
}