package org.wildfly.swarm.config.messaging.activemq.server;

import org.wildfly.swarm.config.runtime.Address;
import org.wildfly.swarm.config.runtime.ResourceType;
import java.beans.PropertyChangeSupport;
import java.beans.PropertyChangeListener;
import org.wildfly.swarm.config.runtime.ModelNodeBinding;
import java.util.List;
import org.wildfly.swarm.config.runtime.Subresource;
import org.wildfly.swarm.config.messaging.activemq.server.RoleConsumer;
import org.wildfly.swarm.config.messaging.activemq.server.RoleSupplier;
import org.wildfly.swarm.config.messaging.activemq.server.Role;
/**
 * A security setting allows sets of permissions to be defined against queues
 * based on their address.
 */
@Address("/subsystem=messaging-activemq/server=*/security-setting=*")
@ResourceType("security-setting")
public class SecuritySetting<T extends SecuritySetting<T>> {

	private String key;
	private PropertyChangeSupport pcs;
	private SecuritySettingResources subresources = new SecuritySettingResources();

	public SecuritySetting(String key) {
		this.key = key;
	}

	public String getKey() {
		return this.key;
	}

	/**
	 * Adds a property change listener
	 */
	public void addPropertyChangeListener(PropertyChangeListener listener) {
		if (null == this.pcs)
			this.pcs = new PropertyChangeSupport(this);
		this.pcs.addPropertyChangeListener(listener);
	}

	/**
	 * Removes a property change listener
	 */
	public void removePropertyChangeListener(PropertyChangeListener listener) {
		if (this.pcs != null)
			this.pcs.removePropertyChangeListener(listener);
	}

	public SecuritySettingResources subresources() {
		return this.subresources;
	}

	/**
	 * Add all Role objects to this subresource
	 * 
	 * @return this
	 * @param value
	 *            List of Role objects.
	 */
	@SuppressWarnings("unchecked")
	public T roles(List<Role> value) {
		this.subresources.roles = value;
		return (T) this;
	}

	/**
	 * Add the Role object to the list of subresources
	 * 
	 * @param value
	 *            The Role to add
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T role(Role value) {
		this.subresources.roles.add(value);
		return (T) this;
	}

	/**
	 * Create and configure a Role object to the list of subresources
	 * 
	 * @param key
	 *            The key for the Role resource
	 * @param config
	 *            The RoleConsumer to use
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T role(String childKey, RoleConsumer consumer) {
		Role<? extends Role> child = new Role<>(childKey);
		if (consumer != null) {
			consumer.accept(child);
		}
		role(child);
		return (T) this;
	}

	/**
	 * Create and configure a Role object to the list of subresources
	 * 
	 * @param key
	 *            The key for the Role resource
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T role(String childKey) {
		role(childKey, null);
		return (T) this;
	}

	/**
	 * Install a supplied Role object to the list of subresources
	 */
	@SuppressWarnings("unchecked")
	public T role(RoleSupplier supplier) {
		role(supplier.get());
		return (T) this;
	}

	/**
	 * Child mutators for SecuritySetting
	 */
	public static class SecuritySettingResources {
		/**
		 * A security role.
		 */
		private List<Role> roles = new java.util.ArrayList<>();

		/**
		 * Get the list of Role resources
		 * 
		 * @return the list of resources
		 */
		@Subresource
		public List<Role> roles() {
			return this.roles;
		}

		public Role role(String key) {
			return this.roles.stream().filter(e -> e.getKey().equals(key))
					.findFirst().orElse(null);
		}
	}
}