package org.wildfly.swarm.config.remoting;

import org.wildfly.swarm.config.runtime.Address;
import org.wildfly.swarm.config.runtime.ResourceType;
import java.beans.PropertyChangeSupport;
import java.beans.PropertyChangeListener;
import org.wildfly.swarm.config.runtime.ModelNodeBinding;
import java.util.List;
import org.wildfly.swarm.config.runtime.Subresource;
import org.wildfly.swarm.config.remoting.PropertyConsumer;
import org.wildfly.swarm.config.remoting.PropertySupplier;
import org.wildfly.swarm.config.remoting.Property;
/**
 * Remoting outbound connection.
 */
@Address("/subsystem=remoting/outbound-connection=*")
@ResourceType("outbound-connection")
public class OutboundConnection<T extends OutboundConnection<T>> {

	private String key;
	private PropertyChangeSupport pcs;
	private String uri;
	private OutboundConnectionResources subresources = new OutboundConnectionResources();

	public OutboundConnection(String key) {
		this.key = key;
	}

	public String getKey() {
		return this.key;
	}

	/**
	 * Adds a property change listener
	 */
	public void addPropertyChangeListener(PropertyChangeListener listener) {
		if (null == this.pcs)
			this.pcs = new PropertyChangeSupport(this);
		this.pcs.addPropertyChangeListener(listener);
	}

	/**
	 * Removes a property change listener
	 */
	public void removePropertyChangeListener(PropertyChangeListener listener) {
		if (this.pcs != null)
			this.pcs.removePropertyChangeListener(listener);
	}

	/**
	 * The connection URI for the outbound connection.
	 */
	@ModelNodeBinding(detypedName = "uri")
	public String uri() {
		return this.uri;
	}

	/**
	 * The connection URI for the outbound connection.
	 */
	@SuppressWarnings("unchecked")
	public T uri(String value) {
		Object oldValue = this.uri;
		this.uri = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("uri", oldValue, value);
		return (T) this;
	}

	public OutboundConnectionResources subresources() {
		return this.subresources;
	}

	/**
	 * Add all Property objects to this subresource
	 * 
	 * @return this
	 * @param value
	 *            List of Property objects.
	 */
	@SuppressWarnings("unchecked")
	public T properties(List<Property> value) {
		this.subresources.properties = value;
		return (T) this;
	}

	/**
	 * Add the Property object to the list of subresources
	 * 
	 * @param value
	 *            The Property to add
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T property(Property value) {
		this.subresources.properties.add(value);
		return (T) this;
	}

	/**
	 * Create and configure a Property object to the list of subresources
	 * 
	 * @param key
	 *            The key for the Property resource
	 * @param config
	 *            The PropertyConsumer to use
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T property(String childKey, PropertyConsumer consumer) {
		Property<? extends Property> child = new Property<>(childKey);
		if (consumer != null) {
			consumer.accept(child);
		}
		property(child);
		return (T) this;
	}

	/**
	 * Create and configure a Property object to the list of subresources
	 * 
	 * @param key
	 *            The key for the Property resource
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T property(String childKey) {
		property(childKey, null);
		return (T) this;
	}

	/**
	 * Install a supplied Property object to the list of subresources
	 */
	@SuppressWarnings("unchecked")
	public T property(PropertySupplier supplier) {
		property(supplier.get());
		return (T) this;
	}

	/**
	 * Child mutators for OutboundConnection
	 */
	public static class OutboundConnectionResources {
		/**
		 * Properties supported by the underlying provider. The property name is
		 * inferred from the last element of the properties address.
		 */
		private List<Property> properties = new java.util.ArrayList<>();

		/**
		 * Get the list of Property resources
		 * 
		 * @return the list of resources
		 */
		@Subresource
		public List<Property> properties() {
			return this.properties;
		}

		public Property property(String key) {
			return this.properties.stream().filter(e -> e.getKey().equals(key))
					.findFirst().orElse(null);
		}
	}
}