package org.wildfly.swarm.config.logging;

import org.wildfly.swarm.config.runtime.Addresses;
import org.wildfly.swarm.config.runtime.ResourceType;
import java.beans.PropertyChangeSupport;
import java.beans.PropertyChangeListener;
import org.wildfly.swarm.config.runtime.ModelNodeBinding;
import java.util.Arrays;
import org.wildfly.swarm.config.logging.Level;
import java.util.Map;
/**
 * Defines a custom logging handler. The custom handler must extend
 * java.util.logging.Handler.
 */
@Addresses({"/subsystem=logging/logging-profile=*/custom-handler=*",
		"/subsystem=logging/custom-handler=*"})
@ResourceType("custom-handler")
public class CustomHandler<T extends CustomHandler<T>> {

	private String key;
	private PropertyChangeSupport pcs;
	private String attributeClass;
	private Boolean enabled;
	private String encoding;
	private String filterSpec;
	private String formatter;
	private Level level;
	private String module;
	private String namedFormatter;
	private Map properties;

	public CustomHandler(String key) {
		this.key = key;
	}

	public String getKey() {
		return this.key;
	}

	/**
	 * Adds a property change listener
	 */
	public void addPropertyChangeListener(PropertyChangeListener listener) {
		if (null == this.pcs)
			this.pcs = new PropertyChangeSupport(this);
		this.pcs.addPropertyChangeListener(listener);
	}

	/**
	 * Removes a property change listener
	 */
	public void removePropertyChangeListener(PropertyChangeListener listener) {
		if (this.pcs != null)
			this.pcs.removePropertyChangeListener(listener);
	}

	/**
	 * The logging handler class to be used.
	 */
	@ModelNodeBinding(detypedName = "class")
	public String attributeClass() {
		return this.attributeClass;
	}

	/**
	 * The logging handler class to be used.
	 */
	@SuppressWarnings("unchecked")
	public T attributeClass(String value) {
		Object oldValue = this.attributeClass;
		this.attributeClass = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("attributeClass", oldValue, value);
		return (T) this;
	}

	/**
	 * If set to true the handler is enabled and functioning as normal, if set
	 * to false the handler is ignored when processing log messages.
	 */
	@ModelNodeBinding(detypedName = "enabled")
	public Boolean enabled() {
		return this.enabled;
	}

	/**
	 * If set to true the handler is enabled and functioning as normal, if set
	 * to false the handler is ignored when processing log messages.
	 */
	@SuppressWarnings("unchecked")
	public T enabled(Boolean value) {
		Object oldValue = this.enabled;
		this.enabled = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("enabled", oldValue, value);
		return (T) this;
	}

	/**
	 * The character encoding used by this Handler.
	 */
	@ModelNodeBinding(detypedName = "encoding")
	public String encoding() {
		return this.encoding;
	}

	/**
	 * The character encoding used by this Handler.
	 */
	@SuppressWarnings("unchecked")
	public T encoding(String value) {
		Object oldValue = this.encoding;
		this.encoding = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("encoding", oldValue, value);
		return (T) this;
	}

	/**
	 * A filter expression value to define a filter. Example for a filter that
	 * does not match a pattern: not(match("JBAS.*"))
	 */
	@ModelNodeBinding(detypedName = "filter-spec")
	public String filterSpec() {
		return this.filterSpec;
	}

	/**
	 * A filter expression value to define a filter. Example for a filter that
	 * does not match a pattern: not(match("JBAS.*"))
	 */
	@SuppressWarnings("unchecked")
	public T filterSpec(String value) {
		Object oldValue = this.filterSpec;
		this.filterSpec = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("filterSpec", oldValue, value);
		return (T) this;
	}

	/**
	 * Defines a pattern for the formatter.
	 */
	@ModelNodeBinding(detypedName = "formatter")
	public String formatter() {
		return this.formatter;
	}

	/**
	 * Defines a pattern for the formatter.
	 */
	@SuppressWarnings("unchecked")
	public T formatter(String value) {
		Object oldValue = this.formatter;
		this.formatter = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("formatter", oldValue, value);
		return (T) this;
	}

	@Deprecated
	@SuppressWarnings("unchecked")
	public T level(final String level) {
		if (level == null) {
			this.level = null;
		} else {
			boolean found = false;
			for (Level e : Level.values()) {
				if (e.toString().equals(level)) {
					level(e);
					found = true;
					break;
				}
			}
			if (!found)
				throw new RuntimeException(String.format(
						"Value '%s' not valid. Valid values are: %s", level,
						Arrays.asList(Level.values())));
		}
		return (T) this;
	}

	/**
	 * The log level specifying which message levels will be logged by this
	 * logger. Message levels lower than this value will be discarded.
	 */
	@ModelNodeBinding(detypedName = "level")
	public Level level() {
		return this.level;
	}

	/**
	 * The log level specifying which message levels will be logged by this
	 * logger. Message levels lower than this value will be discarded.
	 */
	@SuppressWarnings("unchecked")
	public T level(Level value) {
		Object oldValue = this.level;
		this.level = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("level", oldValue, value);
		return (T) this;
	}

	/**
	 * The module that the logging handler depends on.
	 */
	@ModelNodeBinding(detypedName = "module")
	public String module() {
		return this.module;
	}

	/**
	 * The module that the logging handler depends on.
	 */
	@SuppressWarnings("unchecked")
	public T module(String value) {
		Object oldValue = this.module;
		this.module = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("module", oldValue, value);
		return (T) this;
	}

	/**
	 * The name of the defined formatter to be used on the handler.
	 */
	@ModelNodeBinding(detypedName = "named-formatter")
	public String namedFormatter() {
		return this.namedFormatter;
	}

	/**
	 * The name of the defined formatter to be used on the handler.
	 */
	@SuppressWarnings("unchecked")
	public T namedFormatter(String value) {
		Object oldValue = this.namedFormatter;
		this.namedFormatter = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("namedFormatter", oldValue, value);
		return (T) this;
	}

	/**
	 * Defines the properties used for the logging handler. All properties must
	 * be accessible via a setter method.
	 */
	@ModelNodeBinding(detypedName = "properties")
	public Map properties() {
		return this.properties;
	}

	/**
	 * Defines the properties used for the logging handler. All properties must
	 * be accessible via a setter method.
	 */
	@SuppressWarnings("unchecked")
	public T properties(Map value) {
		Object oldValue = this.properties;
		this.properties = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("properties", oldValue, value);
		return (T) this;
	}

	/**
	 * Defines the properties used for the logging handler. All properties must
	 * be accessible via a setter method.
	 */
	@SuppressWarnings("unchecked")
	public T property(String key, Object value) {
		if (this.properties == null) {
			this.properties = new java.util.HashMap<>();
		}
		this.properties.put(key, value);
		return (T) this;
	}
}