package org.wildfly.swarm.config.messaging.activemq.server;

import org.wildfly.swarm.config.runtime.Address;
import org.wildfly.swarm.config.runtime.ResourceType;
import java.beans.PropertyChangeSupport;
import java.beans.PropertyChangeListener;
import org.wildfly.swarm.config.runtime.ModelNodeBinding;
/**
 * Multicast group to listen to receive broadcast from other servers announcing
 * their connectors.
 */
@Address("/subsystem=messaging-activemq/server=*/discovery-group=*")
@ResourceType("discovery-group")
public class DiscoveryGroup<T extends DiscoveryGroup<T>> {

	private String key;
	private PropertyChangeSupport pcs;
	private Long initialWaitTimeout;
	private String jgroupsChannel;
	private String jgroupsStack;
	private Long refreshTimeout;
	private String socketBinding;

	public DiscoveryGroup(String key) {
		this.key = key;
	}

	public String getKey() {
		return this.key;
	}

	/**
	 * Adds a property change listener
	 */
	public void addPropertyChangeListener(PropertyChangeListener listener) {
		if (null == this.pcs)
			this.pcs = new PropertyChangeSupport(this);
		this.pcs.addPropertyChangeListener(listener);
	}

	/**
	 * Removes a property change listener
	 */
	public void removePropertyChangeListener(PropertyChangeListener listener) {
		if (this.pcs != null)
			this.pcs.removePropertyChangeListener(listener);
	}

	/**
	 * Period, in ms, to wait for an initial broadcast to give us at least one
	 * node in the cluster.
	 */
	@ModelNodeBinding(detypedName = "initial-wait-timeout")
	public Long initialWaitTimeout() {
		return this.initialWaitTimeout;
	}

	/**
	 * Period, in ms, to wait for an initial broadcast to give us at least one
	 * node in the cluster.
	 */
	@SuppressWarnings("unchecked")
	public T initialWaitTimeout(Long value) {
		Object oldValue = this.initialWaitTimeout;
		this.initialWaitTimeout = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("initialWaitTimeout", oldValue, value);
		return (T) this;
	}

	/**
	 * The name used by a JGroups channel to join a cluster.
	 */
	@ModelNodeBinding(detypedName = "jgroups-channel")
	public String jgroupsChannel() {
		return this.jgroupsChannel;
	}

	/**
	 * The name used by a JGroups channel to join a cluster.
	 */
	@SuppressWarnings("unchecked")
	public T jgroupsChannel(String value) {
		Object oldValue = this.jgroupsChannel;
		this.jgroupsChannel = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("jgroupsChannel", oldValue, value);
		return (T) this;
	}

	/**
	 * The name of a stack defined in the org.jboss.as.clustering.jgroups
	 * subsystem that is used to form a cluster.
	 */
	@ModelNodeBinding(detypedName = "jgroups-stack")
	public String jgroupsStack() {
		return this.jgroupsStack;
	}

	/**
	 * The name of a stack defined in the org.jboss.as.clustering.jgroups
	 * subsystem that is used to form a cluster.
	 */
	@SuppressWarnings("unchecked")
	public T jgroupsStack(String value) {
		Object oldValue = this.jgroupsStack;
		this.jgroupsStack = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("jgroupsStack", oldValue, value);
		return (T) this;
	}

	/**
	 * Period the discovery group waits after receiving the last broadcast from
	 * a particular server before removing that server's connector pair entry
	 * from its list.
	 */
	@ModelNodeBinding(detypedName = "refresh-timeout")
	public Long refreshTimeout() {
		return this.refreshTimeout;
	}

	/**
	 * Period the discovery group waits after receiving the last broadcast from
	 * a particular server before removing that server's connector pair entry
	 * from its list.
	 */
	@SuppressWarnings("unchecked")
	public T refreshTimeout(Long value) {
		Object oldValue = this.refreshTimeout;
		this.refreshTimeout = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("refreshTimeout", oldValue, value);
		return (T) this;
	}

	/**
	 * The discovery group socket binding.
	 */
	@ModelNodeBinding(detypedName = "socket-binding")
	public String socketBinding() {
		return this.socketBinding;
	}

	/**
	 * The discovery group socket binding.
	 */
	@SuppressWarnings("unchecked")
	public T socketBinding(String value) {
		Object oldValue = this.socketBinding;
		this.socketBinding = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("socketBinding", oldValue, value);
		return (T) this;
	}
}