package org.wildfly.swarm.config.undertow.server;

import org.wildfly.swarm.config.runtime.Address;
import org.wildfly.swarm.config.runtime.ResourceType;
import java.beans.PropertyChangeSupport;
import java.beans.PropertyChangeListener;
import org.wildfly.swarm.config.runtime.ModelNodeBinding;
import java.util.List;
import java.util.Arrays;
/**
 * http listener
 */
@Address("/subsystem=undertow/server=*/https-listener=*")
@ResourceType("https-listener")
public class HttpsListener<T extends HttpsListener<T>> {

	private String key;
	private PropertyChangeSupport pcs;
	private Boolean allowEncodedSlash;
	private Boolean allowEqualsInCookieValue;
	private Boolean alwaysSetKeepAlive;
	private Boolean bufferPipelinedData;
	private String bufferPool;
	private Long bytesReceived;
	private Long bytesSent;
	private Boolean decodeUrl;
	private List<String> disallowedMethods;
	private Boolean enableHttp2;
	private Boolean enableSpdy;
	private Boolean enabled;
	private String enabledCipherSuites;
	private String enabledProtocols;
	private Long errorCount;
	private Integer maxBufferedRequestSize;
	private Integer maxConnections;
	private Integer maxCookies;
	private Integer maxHeaderSize;
	private Integer maxHeaders;
	private Integer maxParameters;
	private Long maxPostSize;
	private Long maxProcessingTime;
	private Integer noRequestTimeout;
	private Long processingTime;
	private Integer readTimeout;
	private Integer receiveBuffer;
	private Boolean recordRequestStartTime;
	private Long requestCount;
	private Integer requestParseTimeout;
	private Boolean resolvePeerAddress;
	private Boolean secure;
	private String securityRealm;
	private Integer sendBuffer;
	private String socketBinding;
	private Integer sslSessionCacheSize;
	private Integer sslSessionTimeout;
	private Integer tcpBacklog;
	private Boolean tcpKeepAlive;
	private String urlCharset;
	private VerifyClient verifyClient;
	private String worker;
	private Integer writeTimeout;

	public HttpsListener(String key) {
		this.key = key;
	}

	public String getKey() {
		return this.key;
	}

	/**
	 * Adds a property change listener
	 */
	public void addPropertyChangeListener(PropertyChangeListener listener) {
		if (null == this.pcs)
			this.pcs = new PropertyChangeSupport(this);
		this.pcs.addPropertyChangeListener(listener);
	}

	/**
	 * Removes a property change listener
	 */
	public void removePropertyChangeListener(PropertyChangeListener listener) {
		if (this.pcs != null)
			this.pcs.removePropertyChangeListener(listener);
	}

	/**
	 * If a request comes in with encoded / characters (i.e. %2F), will these be
	 * decoded.
	 */
	@ModelNodeBinding(detypedName = "allow-encoded-slash")
	public Boolean allowEncodedSlash() {
		return this.allowEncodedSlash;
	}

	/**
	 * If a request comes in with encoded / characters (i.e. %2F), will these be
	 * decoded.
	 */
	@SuppressWarnings("unchecked")
	public T allowEncodedSlash(Boolean value) {
		Object oldValue = this.allowEncodedSlash;
		this.allowEncodedSlash = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("allowEncodedSlash", oldValue, value);
		return (T) this;
	}

	/**
	 * If this is true then Undertow will allow non-escaped equals characters in
	 * unquoted cookie values. Unquoted cookie values may not contain equals
	 * characters. If present the value ends before the equals sign. The
	 * remainder of the cookie value will be dropped.
	 */
	@ModelNodeBinding(detypedName = "allow-equals-in-cookie-value")
	public Boolean allowEqualsInCookieValue() {
		return this.allowEqualsInCookieValue;
	}

	/**
	 * If this is true then Undertow will allow non-escaped equals characters in
	 * unquoted cookie values. Unquoted cookie values may not contain equals
	 * characters. If present the value ends before the equals sign. The
	 * remainder of the cookie value will be dropped.
	 */
	@SuppressWarnings("unchecked")
	public T allowEqualsInCookieValue(Boolean value) {
		Object oldValue = this.allowEqualsInCookieValue;
		this.allowEqualsInCookieValue = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("allowEqualsInCookieValue", oldValue,
					value);
		return (T) this;
	}

	/**
	 * If this is true then a Connection: keep-alive header will be added to
	 * responses, even when it is not strictly required by the specification.
	 */
	@ModelNodeBinding(detypedName = "always-set-keep-alive")
	public Boolean alwaysSetKeepAlive() {
		return this.alwaysSetKeepAlive;
	}

	/**
	 * If this is true then a Connection: keep-alive header will be added to
	 * responses, even when it is not strictly required by the specification.
	 */
	@SuppressWarnings("unchecked")
	public T alwaysSetKeepAlive(Boolean value) {
		Object oldValue = this.alwaysSetKeepAlive;
		this.alwaysSetKeepAlive = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("alwaysSetKeepAlive", oldValue, value);
		return (T) this;
	}

	/**
	 * If we should buffer pipelined requests.
	 */
	@ModelNodeBinding(detypedName = "buffer-pipelined-data")
	public Boolean bufferPipelinedData() {
		return this.bufferPipelinedData;
	}

	/**
	 * If we should buffer pipelined requests.
	 */
	@SuppressWarnings("unchecked")
	public T bufferPipelinedData(Boolean value) {
		Object oldValue = this.bufferPipelinedData;
		this.bufferPipelinedData = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("bufferPipelinedData", oldValue, value);
		return (T) this;
	}

	/**
	 * The AJP listeners buffer pool
	 */
	@ModelNodeBinding(detypedName = "buffer-pool")
	public String bufferPool() {
		return this.bufferPool;
	}

	/**
	 * The AJP listeners buffer pool
	 */
	@SuppressWarnings("unchecked")
	public T bufferPool(String value) {
		Object oldValue = this.bufferPool;
		this.bufferPool = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("bufferPool", oldValue, value);
		return (T) this;
	}

	/**
	 * The number of bytes that have been received by this listener
	 */
	@ModelNodeBinding(detypedName = "bytes-received")
	public Long bytesReceived() {
		return this.bytesReceived;
	}

	/**
	 * The number of bytes that have been received by this listener
	 */
	@SuppressWarnings("unchecked")
	public T bytesReceived(Long value) {
		Object oldValue = this.bytesReceived;
		this.bytesReceived = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("bytesReceived", oldValue, value);
		return (T) this;
	}

	/**
	 * The number of bytes that have been sent out on this listener
	 */
	@ModelNodeBinding(detypedName = "bytes-sent")
	public Long bytesSent() {
		return this.bytesSent;
	}

	/**
	 * The number of bytes that have been sent out on this listener
	 */
	@SuppressWarnings("unchecked")
	public T bytesSent(Long value) {
		Object oldValue = this.bytesSent;
		this.bytesSent = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("bytesSent", oldValue, value);
		return (T) this;
	}

	/**
	 * If this is true then the parser will decode the URL and query parameters
	 * using the selected character encoding (UTF-8 by default). If this is
	 * false they will not be decoded. This will allow a later handler to decode
	 * them into whatever charset is desired.
	 */
	@ModelNodeBinding(detypedName = "decode-url")
	public Boolean decodeUrl() {
		return this.decodeUrl;
	}

	/**
	 * If this is true then the parser will decode the URL and query parameters
	 * using the selected character encoding (UTF-8 by default). If this is
	 * false they will not be decoded. This will allow a later handler to decode
	 * them into whatever charset is desired.
	 */
	@SuppressWarnings("unchecked")
	public T decodeUrl(Boolean value) {
		Object oldValue = this.decodeUrl;
		this.decodeUrl = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("decodeUrl", oldValue, value);
		return (T) this;
	}

	/**
	 * A comma separated list of HTTP methods that are not allowed
	 */
	@ModelNodeBinding(detypedName = "disallowed-methods")
	public List<String> disallowedMethods() {
		return this.disallowedMethods;
	}

	/**
	 * A comma separated list of HTTP methods that are not allowed
	 */
	@SuppressWarnings("unchecked")
	public T disallowedMethods(List<String> value) {
		Object oldValue = this.disallowedMethods;
		this.disallowedMethods = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("disallowedMethods", oldValue, value);
		return (T) this;
	}

	/**
	 * A comma separated list of HTTP methods that are not allowed
	 */
	@SuppressWarnings("unchecked")
	public T disallowedMethod(String value) {
		if (this.disallowedMethods == null) {
			this.disallowedMethods = new java.util.ArrayList<>();
		}
		this.disallowedMethods.add(value);
		return (T) this;
	}

	/**
	 * A comma separated list of HTTP methods that are not allowed
	 */
	@SuppressWarnings("unchecked")
	public T disallowedMethods(String... args) {
		disallowedMethods(Arrays.asList(args));
		return (T) this;
	}

	/**
	 * Enables HTTP2 support for this listener
	 */
	@ModelNodeBinding(detypedName = "enable-http2")
	public Boolean enableHttp2() {
		return this.enableHttp2;
	}

	/**
	 * Enables HTTP2 support for this listener
	 */
	@SuppressWarnings("unchecked")
	public T enableHttp2(Boolean value) {
		Object oldValue = this.enableHttp2;
		this.enableHttp2 = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("enableHttp2", oldValue, value);
		return (T) this;
	}

	/**
	 * Enables SPDY support for this listener
	 */
	@ModelNodeBinding(detypedName = "enable-spdy")
	public Boolean enableSpdy() {
		return this.enableSpdy;
	}

	/**
	 * Enables SPDY support for this listener
	 */
	@SuppressWarnings("unchecked")
	public T enableSpdy(Boolean value) {
		Object oldValue = this.enableSpdy;
		this.enableSpdy = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("enableSpdy", oldValue, value);
		return (T) this;
	}

	/**
	 * If the listener is enabled
	 */
	@ModelNodeBinding(detypedName = "enabled")
	public Boolean enabled() {
		return this.enabled;
	}

	/**
	 * If the listener is enabled
	 */
	@SuppressWarnings("unchecked")
	public T enabled(Boolean value) {
		Object oldValue = this.enabled;
		this.enabled = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("enabled", oldValue, value);
		return (T) this;
	}

	/**
	 * Configures Enabled SSL cyphers
	 */
	@ModelNodeBinding(detypedName = "enabled-cipher-suites")
	public String enabledCipherSuites() {
		return this.enabledCipherSuites;
	}

	/**
	 * Configures Enabled SSL cyphers
	 */
	@SuppressWarnings("unchecked")
	public T enabledCipherSuites(String value) {
		Object oldValue = this.enabledCipherSuites;
		this.enabledCipherSuites = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("enabledCipherSuites", oldValue, value);
		return (T) this;
	}

	/**
	 * Configures SSL protocols
	 */
	@ModelNodeBinding(detypedName = "enabled-protocols")
	public String enabledProtocols() {
		return this.enabledProtocols;
	}

	/**
	 * Configures SSL protocols
	 */
	@SuppressWarnings("unchecked")
	public T enabledProtocols(String value) {
		Object oldValue = this.enabledProtocols;
		this.enabledProtocols = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("enabledProtocols", oldValue, value);
		return (T) this;
	}

	/**
	 * The number of 500 responses that have been sent by this listener
	 */
	@ModelNodeBinding(detypedName = "error-count")
	public Long errorCount() {
		return this.errorCount;
	}

	/**
	 * The number of 500 responses that have been sent by this listener
	 */
	@SuppressWarnings("unchecked")
	public T errorCount(Long value) {
		Object oldValue = this.errorCount;
		this.errorCount = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("errorCount", oldValue, value);
		return (T) this;
	}

	/**
	 * Maximum size of a buffered request, in bytesRequests are not usually
	 * buffered, the most common case is when performing SSL renegotiation for a
	 * POST request, and the post data must be fullybuffered in order to perform
	 * the renegotiation.
	 */
	@ModelNodeBinding(detypedName = "max-buffered-request-size")
	public Integer maxBufferedRequestSize() {
		return this.maxBufferedRequestSize;
	}

	/**
	 * Maximum size of a buffered request, in bytesRequests are not usually
	 * buffered, the most common case is when performing SSL renegotiation for a
	 * POST request, and the post data must be fullybuffered in order to perform
	 * the renegotiation.
	 */
	@SuppressWarnings("unchecked")
	public T maxBufferedRequestSize(Integer value) {
		Object oldValue = this.maxBufferedRequestSize;
		this.maxBufferedRequestSize = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("maxBufferedRequestSize", oldValue,
					value);
		return (T) this;
	}

	/**
	 * The maximum number of concurrent connections.
	 */
	@ModelNodeBinding(detypedName = "max-connections")
	public Integer maxConnections() {
		return this.maxConnections;
	}

	/**
	 * The maximum number of concurrent connections.
	 */
	@SuppressWarnings("unchecked")
	public T maxConnections(Integer value) {
		Object oldValue = this.maxConnections;
		this.maxConnections = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("maxConnections", oldValue, value);
		return (T) this;
	}

	/**
	 * The maximum number of cookies that will be parsed. This is used to
	 * protect against hash vulnerabilities.
	 */
	@ModelNodeBinding(detypedName = "max-cookies")
	public Integer maxCookies() {
		return this.maxCookies;
	}

	/**
	 * The maximum number of cookies that will be parsed. This is used to
	 * protect against hash vulnerabilities.
	 */
	@SuppressWarnings("unchecked")
	public T maxCookies(Integer value) {
		Object oldValue = this.maxCookies;
		this.maxCookies = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("maxCookies", oldValue, value);
		return (T) this;
	}

	/**
	 * The maximum size in bytes of a http request header.
	 */
	@ModelNodeBinding(detypedName = "max-header-size")
	public Integer maxHeaderSize() {
		return this.maxHeaderSize;
	}

	/**
	 * The maximum size in bytes of a http request header.
	 */
	@SuppressWarnings("unchecked")
	public T maxHeaderSize(Integer value) {
		Object oldValue = this.maxHeaderSize;
		this.maxHeaderSize = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("maxHeaderSize", oldValue, value);
		return (T) this;
	}

	/**
	 * The maximum number of headers that will be parsed. This is used to
	 * protect against hash vulnerabilities.
	 */
	@ModelNodeBinding(detypedName = "max-headers")
	public Integer maxHeaders() {
		return this.maxHeaders;
	}

	/**
	 * The maximum number of headers that will be parsed. This is used to
	 * protect against hash vulnerabilities.
	 */
	@SuppressWarnings("unchecked")
	public T maxHeaders(Integer value) {
		Object oldValue = this.maxHeaders;
		this.maxHeaders = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("maxHeaders", oldValue, value);
		return (T) this;
	}

	/**
	 * The maximum number of parameters that will be parsed. This is used to
	 * protect against hash vulnerabilities.This applies to both query
	 * parameters, and to POST data, but is not cumulative (i.e. you can
	 * potentially have max parameters * 2 total parameters).
	 */
	@ModelNodeBinding(detypedName = "max-parameters")
	public Integer maxParameters() {
		return this.maxParameters;
	}

	/**
	 * The maximum number of parameters that will be parsed. This is used to
	 * protect against hash vulnerabilities.This applies to both query
	 * parameters, and to POST data, but is not cumulative (i.e. you can
	 * potentially have max parameters * 2 total parameters).
	 */
	@SuppressWarnings("unchecked")
	public T maxParameters(Integer value) {
		Object oldValue = this.maxParameters;
		this.maxParameters = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("maxParameters", oldValue, value);
		return (T) this;
	}

	/**
	 * The maximum size of a post that will be accepted
	 */
	@ModelNodeBinding(detypedName = "max-post-size")
	public Long maxPostSize() {
		return this.maxPostSize;
	}

	/**
	 * The maximum size of a post that will be accepted
	 */
	@SuppressWarnings("unchecked")
	public T maxPostSize(Long value) {
		Object oldValue = this.maxPostSize;
		this.maxPostSize = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("maxPostSize", oldValue, value);
		return (T) this;
	}

	/**
	 * The maximum processing time taken by a request on this listener
	 */
	@ModelNodeBinding(detypedName = "max-processing-time")
	public Long maxProcessingTime() {
		return this.maxProcessingTime;
	}

	/**
	 * The maximum processing time taken by a request on this listener
	 */
	@SuppressWarnings("unchecked")
	public T maxProcessingTime(Long value) {
		Object oldValue = this.maxProcessingTime;
		this.maxProcessingTime = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("maxProcessingTime", oldValue, value);
		return (T) this;
	}

	/**
	 * The length of time in milliseconds that the connection can be idle before
	 * it is closed by the container
	 */
	@ModelNodeBinding(detypedName = "no-request-timeout")
	public Integer noRequestTimeout() {
		return this.noRequestTimeout;
	}

	/**
	 * The length of time in milliseconds that the connection can be idle before
	 * it is closed by the container
	 */
	@SuppressWarnings("unchecked")
	public T noRequestTimeout(Integer value) {
		Object oldValue = this.noRequestTimeout;
		this.noRequestTimeout = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("noRequestTimeout", oldValue, value);
		return (T) this;
	}

	/**
	 * The total processing time of all requests handed by this listener
	 */
	@ModelNodeBinding(detypedName = "processing-time")
	public Long processingTime() {
		return this.processingTime;
	}

	/**
	 * The total processing time of all requests handed by this listener
	 */
	@SuppressWarnings("unchecked")
	public T processingTime(Long value) {
		Object oldValue = this.processingTime;
		this.processingTime = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("processingTime", oldValue, value);
		return (T) this;
	}

	/**
	 * Configure a read timeout for a socket, in milliseconds. If the given
	 * amount of time elapses without a successful read taking place, the
	 * socket's next read will throw a {@link ReadTimeoutException}.
	 */
	@ModelNodeBinding(detypedName = "read-timeout")
	public Integer readTimeout() {
		return this.readTimeout;
	}

	/**
	 * Configure a read timeout for a socket, in milliseconds. If the given
	 * amount of time elapses without a successful read taking place, the
	 * socket's next read will throw a {@link ReadTimeoutException}.
	 */
	@SuppressWarnings("unchecked")
	public T readTimeout(Integer value) {
		Object oldValue = this.readTimeout;
		this.readTimeout = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("readTimeout", oldValue, value);
		return (T) this;
	}

	/**
	 * The receive buffer size.
	 */
	@ModelNodeBinding(detypedName = "receive-buffer")
	public Integer receiveBuffer() {
		return this.receiveBuffer;
	}

	/**
	 * The receive buffer size.
	 */
	@SuppressWarnings("unchecked")
	public T receiveBuffer(Integer value) {
		Object oldValue = this.receiveBuffer;
		this.receiveBuffer = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("receiveBuffer", oldValue, value);
		return (T) this;
	}

	/**
	 * If this is true then Undertow will record the request start time, to
	 * allow for request time to be logged. This has a small but measurable
	 * performance impact
	 */
	@ModelNodeBinding(detypedName = "record-request-start-time")
	public Boolean recordRequestStartTime() {
		return this.recordRequestStartTime;
	}

	/**
	 * If this is true then Undertow will record the request start time, to
	 * allow for request time to be logged. This has a small but measurable
	 * performance impact
	 */
	@SuppressWarnings("unchecked")
	public T recordRequestStartTime(Boolean value) {
		Object oldValue = this.recordRequestStartTime;
		this.recordRequestStartTime = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("recordRequestStartTime", oldValue,
					value);
		return (T) this;
	}

	/**
	 * The number of requests this listener has served
	 */
	@ModelNodeBinding(detypedName = "request-count")
	public Long requestCount() {
		return this.requestCount;
	}

	/**
	 * The number of requests this listener has served
	 */
	@SuppressWarnings("unchecked")
	public T requestCount(Long value) {
		Object oldValue = this.requestCount;
		this.requestCount = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("requestCount", oldValue, value);
		return (T) this;
	}

	/**
	 * The maximum amount of time (in milliseconds) that can be spent parsing
	 * the request
	 */
	@ModelNodeBinding(detypedName = "request-parse-timeout")
	public Integer requestParseTimeout() {
		return this.requestParseTimeout;
	}

	/**
	 * The maximum amount of time (in milliseconds) that can be spent parsing
	 * the request
	 */
	@SuppressWarnings("unchecked")
	public T requestParseTimeout(Integer value) {
		Object oldValue = this.requestParseTimeout;
		this.requestParseTimeout = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("requestParseTimeout", oldValue, value);
		return (T) this;
	}

	/**
	 * Enables host dns lookup
	 */
	@ModelNodeBinding(detypedName = "resolve-peer-address")
	public Boolean resolvePeerAddress() {
		return this.resolvePeerAddress;
	}

	/**
	 * Enables host dns lookup
	 */
	@SuppressWarnings("unchecked")
	public T resolvePeerAddress(Boolean value) {
		Object oldValue = this.resolvePeerAddress;
		this.resolvePeerAddress = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("resolvePeerAddress", oldValue, value);
		return (T) this;
	}

	/**
	 * If this is true then requests that originate from this listener are
	 * marked as secure, even if the request is not using HTTPS.
	 */
	@ModelNodeBinding(detypedName = "secure")
	public Boolean secure() {
		return this.secure;
	}

	/**
	 * If this is true then requests that originate from this listener are
	 * marked as secure, even if the request is not using HTTPS.
	 */
	@SuppressWarnings("unchecked")
	public T secure(Boolean value) {
		Object oldValue = this.secure;
		this.secure = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("secure", oldValue, value);
		return (T) this;
	}

	/**
	 * The listeners security realm
	 */
	@ModelNodeBinding(detypedName = "security-realm")
	public String securityRealm() {
		return this.securityRealm;
	}

	/**
	 * The listeners security realm
	 */
	@SuppressWarnings("unchecked")
	public T securityRealm(String value) {
		Object oldValue = this.securityRealm;
		this.securityRealm = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("securityRealm", oldValue, value);
		return (T) this;
	}

	/**
	 * The send buffer size.
	 */
	@ModelNodeBinding(detypedName = "send-buffer")
	public Integer sendBuffer() {
		return this.sendBuffer;
	}

	/**
	 * The send buffer size.
	 */
	@SuppressWarnings("unchecked")
	public T sendBuffer(Integer value) {
		Object oldValue = this.sendBuffer;
		this.sendBuffer = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("sendBuffer", oldValue, value);
		return (T) this;
	}

	/**
	 * The AJP listener socket binder
	 */
	@ModelNodeBinding(detypedName = "socket-binding")
	public String socketBinding() {
		return this.socketBinding;
	}

	/**
	 * The AJP listener socket binder
	 */
	@SuppressWarnings("unchecked")
	public T socketBinding(String value) {
		Object oldValue = this.socketBinding;
		this.socketBinding = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("socketBinding", oldValue, value);
		return (T) this;
	}

	/**
	 * The maximum number of active SSL sessions
	 */
	@ModelNodeBinding(detypedName = "ssl-session-cache-size")
	public Integer sslSessionCacheSize() {
		return this.sslSessionCacheSize;
	}

	/**
	 * The maximum number of active SSL sessions
	 */
	@SuppressWarnings("unchecked")
	public T sslSessionCacheSize(Integer value) {
		Object oldValue = this.sslSessionCacheSize;
		this.sslSessionCacheSize = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("sslSessionCacheSize", oldValue, value);
		return (T) this;
	}

	/**
	 * The timeout for SSL sessions, in seconds
	 */
	@ModelNodeBinding(detypedName = "ssl-session-timeout")
	public Integer sslSessionTimeout() {
		return this.sslSessionTimeout;
	}

	/**
	 * The timeout for SSL sessions, in seconds
	 */
	@SuppressWarnings("unchecked")
	public T sslSessionTimeout(Integer value) {
		Object oldValue = this.sslSessionTimeout;
		this.sslSessionTimeout = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("sslSessionTimeout", oldValue, value);
		return (T) this;
	}

	/**
	 * Configure a server with the specified backlog.
	 */
	@ModelNodeBinding(detypedName = "tcp-backlog")
	public Integer tcpBacklog() {
		return this.tcpBacklog;
	}

	/**
	 * Configure a server with the specified backlog.
	 */
	@SuppressWarnings("unchecked")
	public T tcpBacklog(Integer value) {
		Object oldValue = this.tcpBacklog;
		this.tcpBacklog = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("tcpBacklog", oldValue, value);
		return (T) this;
	}

	/**
	 * Configure a channel to send TCP keep-alive messages in an
	 * implementation-dependent manner.
	 */
	@ModelNodeBinding(detypedName = "tcp-keep-alive")
	public Boolean tcpKeepAlive() {
		return this.tcpKeepAlive;
	}

	/**
	 * Configure a channel to send TCP keep-alive messages in an
	 * implementation-dependent manner.
	 */
	@SuppressWarnings("unchecked")
	public T tcpKeepAlive(Boolean value) {
		Object oldValue = this.tcpKeepAlive;
		this.tcpKeepAlive = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("tcpKeepAlive", oldValue, value);
		return (T) this;
	}

	/**
	 * URL charset
	 */
	@ModelNodeBinding(detypedName = "url-charset")
	public String urlCharset() {
		return this.urlCharset;
	}

	/**
	 * URL charset
	 */
	@SuppressWarnings("unchecked")
	public T urlCharset(String value) {
		Object oldValue = this.urlCharset;
		this.urlCharset = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("urlCharset", oldValue, value);
		return (T) this;
	}

	@Deprecated
	@SuppressWarnings("unchecked")
	public T verifyClient(final String verifyClient) {
		if (verifyClient == null) {
			this.verifyClient = null;
		} else {
			boolean found = false;
			for (VerifyClient e : VerifyClient.values()) {
				if (e.toString().equals(verifyClient)) {
					verifyClient(e);
					found = true;
					break;
				}
			}
			if (!found)
				throw new RuntimeException(String.format(
						"Value '%s' not valid. Valid values are: %s",
						verifyClient, Arrays.asList(VerifyClient.values())));
		}
		return (T) this;
	}

	/**
	 * The desired SSL client authentication mode for SSL channels
	 */
	@ModelNodeBinding(detypedName = "verify-client")
	public VerifyClient verifyClient() {
		return this.verifyClient;
	}

	/**
	 * The desired SSL client authentication mode for SSL channels
	 */
	@SuppressWarnings("unchecked")
	public T verifyClient(VerifyClient value) {
		Object oldValue = this.verifyClient;
		this.verifyClient = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("verifyClient", oldValue, value);
		return (T) this;
	}

	/**
	 * The AJP listeners XNIO worker
	 */
	@ModelNodeBinding(detypedName = "worker")
	public String worker() {
		return this.worker;
	}

	/**
	 * The AJP listeners XNIO worker
	 */
	@SuppressWarnings("unchecked")
	public T worker(String value) {
		Object oldValue = this.worker;
		this.worker = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("worker", oldValue, value);
		return (T) this;
	}

	/**
	 * Configure a write timeout for a socket, in milliseconds. If the given
	 * amount of time elapses without a successful write taking place, the
	 * socket's next write will throw a {@link WriteTimeoutException}.
	 */
	@ModelNodeBinding(detypedName = "write-timeout")
	public Integer writeTimeout() {
		return this.writeTimeout;
	}

	/**
	 * Configure a write timeout for a socket, in milliseconds. If the given
	 * amount of time elapses without a successful write taking place, the
	 * socket's next write will throw a {@link WriteTimeoutException}.
	 */
	@SuppressWarnings("unchecked")
	public T writeTimeout(Integer value) {
		Object oldValue = this.writeTimeout;
		this.writeTimeout = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("writeTimeout", oldValue, value);
		return (T) this;
	}

	public static enum VerifyClient {
		NOT_REQUESTED("NOT_REQUESTED"), REQUESTED("REQUESTED"), REQUIRED(
				"REQUIRED");
		private final String allowedValue;

		/**
		 * Returns the allowed value for the management model.
		 * 
		 * @return the allowed model value
		 */
		public String getAllowedValue() {
			return allowedValue;
		}

		VerifyClient(String allowedValue) {
			this.allowedValue = allowedValue;
		}

		@Override
		public String toString() {
			return allowedValue;
		}
	}
}