package org.wildfly.swarm.config.transactions;

import org.wildfly.swarm.config.runtime.Address;
import org.wildfly.swarm.config.runtime.ResourceType;
import org.wildfly.swarm.config.runtime.Implicit;
import java.beans.PropertyChangeSupport;
import java.beans.PropertyChangeListener;
import org.wildfly.swarm.config.runtime.ModelNodeBinding;
import java.util.List;
import org.wildfly.swarm.config.runtime.Subresource;
import org.wildfly.swarm.config.transactions.log_store.TransactionsConsumer;
import org.wildfly.swarm.config.transactions.log_store.TransactionsSupplier;
import org.wildfly.swarm.config.transactions.log_store.Transactions;
/**
 * Representation of the transaction logging storage mechanism.
 */
@Address("/subsystem=transactions/log-store=log-store")
@ResourceType("log-store")
@Implicit
public class LogStore<T extends LogStore<T>> {

	private String key;
	private PropertyChangeSupport pcs;
	private Boolean exposeAllLogs;
	private String type;
	private LogStoreResources subresources = new LogStoreResources();

	public LogStore() {
		this.key = "log-store";
		this.pcs = new PropertyChangeSupport(this);
	}

	public String getKey() {
		return this.key;
	}

	/**
	 * Adds a property change listener
	 */
	public void addPropertyChangeListener(PropertyChangeListener listener) {
		if (null == this.pcs)
			this.pcs = new PropertyChangeSupport(this);
		this.pcs.addPropertyChangeListener(listener);
	}

	/**
	 * Removes a property change listener
	 */
	public void removePropertyChangeListener(PropertyChangeListener listener) {
		if (this.pcs != null)
			this.pcs.removePropertyChangeListener(listener);
	}

	/**
	 * Whether to expose all logs like orphans etc. By default only a subset of
	 * transaction logs is exposed.
	 */
	@ModelNodeBinding(detypedName = "expose-all-logs")
	public Boolean exposeAllLogs() {
		return this.exposeAllLogs;
	}

	/**
	 * Whether to expose all logs like orphans etc. By default only a subset of
	 * transaction logs is exposed.
	 */
	@SuppressWarnings("unchecked")
	public T exposeAllLogs(Boolean value) {
		Object oldValue = this.exposeAllLogs;
		this.exposeAllLogs = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("exposeAllLogs", oldValue, value);
		return (T) this;
	}

	/**
	 * Specifies the implementation type of the logging store.
	 */
	@ModelNodeBinding(detypedName = "type")
	public String type() {
		return this.type;
	}

	/**
	 * Specifies the implementation type of the logging store.
	 */
	@SuppressWarnings("unchecked")
	public T type(String value) {
		Object oldValue = this.type;
		this.type = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("type", oldValue, value);
		return (T) this;
	}

	public LogStoreResources subresources() {
		return this.subresources;
	}

	/**
	 * Add all Transactions objects to this subresource
	 * 
	 * @return this
	 * @param value
	 *            List of Transactions objects.
	 */
	@SuppressWarnings("unchecked")
	public T transactions(List<Transactions> value) {
		this.subresources.transactions = value;
		return (T) this;
	}

	/**
	 * Add the Transactions object to the list of subresources
	 * 
	 * @param value
	 *            The Transactions to add
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T transactions(Transactions value) {
		this.subresources.transactions.add(value);
		return (T) this;
	}

	/**
	 * Create and configure a Transactions object to the list of subresources
	 * 
	 * @param key
	 *            The key for the Transactions resource
	 * @param config
	 *            The TransactionsConsumer to use
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T transactions(String childKey, TransactionsConsumer consumer) {
		Transactions<? extends Transactions> child = new Transactions<>(
				childKey);
		if (consumer != null) {
			consumer.accept(child);
		}
		transactions(child);
		return (T) this;
	}

	/**
	 * Create and configure a Transactions object to the list of subresources
	 * 
	 * @param key
	 *            The key for the Transactions resource
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T transactions(String childKey) {
		transactions(childKey, null);
		return (T) this;
	}

	/**
	 * Install a supplied Transactions object to the list of subresources
	 */
	@SuppressWarnings("unchecked")
	public T transactions(TransactionsSupplier supplier) {
		transactions(supplier.get());
		return (T) this;
	}

	/**
	 * Child mutators for LogStore
	 */
	public static class LogStoreResources {
		/**
		 * The persistent information that the transaction manager stores for
		 * the purpose of recovering a transaction in the event of failure. The
		 * probe operation will add and remove transactions from the model as
		 * the corresponding real transactions start and finish the prepare and
		 * commit phases. A stuck transaction will remain in the model until
		 * either it is completed or explicitly removed by the delete operation.
		 */
		private List<Transactions> transactions = new java.util.ArrayList<>();

		/**
		 * Get the list of Transactions resources
		 * 
		 * @return the list of resources
		 */
		@Subresource
		public List<Transactions> transactions() {
			return this.transactions;
		}

		public Transactions transactions(String key) {
			return this.transactions.stream()
					.filter(e -> e.getKey().equals(key)).findFirst()
					.orElse(null);
		}
	}
}