package org.wildfly.swarm.config.ejb3;

import org.wildfly.swarm.config.runtime.Address;
import org.wildfly.swarm.config.runtime.ResourceType;
import java.beans.PropertyChangeSupport;
import java.beans.PropertyChangeListener;
import org.wildfly.swarm.config.runtime.ModelNodeBinding;
import java.util.Map;
/**
 * A thread pool executor with an unbounded queue. Such a thread pool has a core
 * size and a queue with no upper bound. When a task is submitted, if the number
 * of running threads is less than the core size, a new thread is created.
 * Otherwise, the task is placed in queue. If too many tasks are allowed to be
 * submitted to this type of executor, an out of memory condition may occur.
 */
@Address("/subsystem=ejb3/thread-pool=*")
@ResourceType("thread-pool")
public class ThreadPool<T extends ThreadPool<T>> {

	private String key;
	private PropertyChangeSupport pcs;
	private Integer activeCount;
	private Integer completedTaskCount;
	private Integer currentThreadCount;
	private Map keepaliveTime;
	private Integer largestThreadCount;
	private Integer maxThreads;
	private String name;
	private Integer queueSize;
	private Integer rejectedCount;
	private Integer taskCount;
	private String threadFactory;

	public ThreadPool(String key) {
		this.key = key;
	}

	public String getKey() {
		return this.key;
	}

	/**
	 * Adds a property change listener
	 */
	public void addPropertyChangeListener(PropertyChangeListener listener) {
		if (null == this.pcs)
			this.pcs = new PropertyChangeSupport(this);
		this.pcs.addPropertyChangeListener(listener);
	}

	/**
	 * Removes a property change listener
	 */
	public void removePropertyChangeListener(PropertyChangeListener listener) {
		if (this.pcs != null)
			this.pcs.removePropertyChangeListener(listener);
	}

	/**
	 * The approximate number of threads that are actively executing tasks.
	 */
	@ModelNodeBinding(detypedName = "active-count")
	public Integer activeCount() {
		return this.activeCount;
	}

	/**
	 * The approximate number of threads that are actively executing tasks.
	 */
	@SuppressWarnings("unchecked")
	public T activeCount(Integer value) {
		Object oldValue = this.activeCount;
		this.activeCount = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("activeCount", oldValue, value);
		return (T) this;
	}

	/**
	 * The approximate total number of tasks that have completed execution.
	 */
	@ModelNodeBinding(detypedName = "completed-task-count")
	public Integer completedTaskCount() {
		return this.completedTaskCount;
	}

	/**
	 * The approximate total number of tasks that have completed execution.
	 */
	@SuppressWarnings("unchecked")
	public T completedTaskCount(Integer value) {
		Object oldValue = this.completedTaskCount;
		this.completedTaskCount = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("completedTaskCount", oldValue, value);
		return (T) this;
	}

	/**
	 * The current number of threads in the pool.
	 */
	@ModelNodeBinding(detypedName = "current-thread-count")
	public Integer currentThreadCount() {
		return this.currentThreadCount;
	}

	/**
	 * The current number of threads in the pool.
	 */
	@SuppressWarnings("unchecked")
	public T currentThreadCount(Integer value) {
		Object oldValue = this.currentThreadCount;
		this.currentThreadCount = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("currentThreadCount", oldValue, value);
		return (T) this;
	}

	/**
	 * Used to specify the amount of time that pool threads should be kept
	 * running when idle; if not specified, threads will run until the executor
	 * is shut down.
	 */
	@ModelNodeBinding(detypedName = "keepalive-time")
	public Map keepaliveTime() {
		return this.keepaliveTime;
	}

	/**
	 * Used to specify the amount of time that pool threads should be kept
	 * running when idle; if not specified, threads will run until the executor
	 * is shut down.
	 */
	@SuppressWarnings("unchecked")
	public T keepaliveTime(Map value) {
		Object oldValue = this.keepaliveTime;
		this.keepaliveTime = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("keepaliveTime", oldValue, value);
		return (T) this;
	}

	/**
	 * Used to specify the amount of time that pool threads should be kept
	 * running when idle; if not specified, threads will run until the executor
	 * is shut down.
	 */
	@SuppressWarnings("unchecked")
	public T keepaliveTime(String key, Object value) {
		if (this.keepaliveTime == null) {
			this.keepaliveTime = new java.util.HashMap<>();
		}
		this.keepaliveTime.put(key, value);
		return (T) this;
	}

	/**
	 * The largest number of threads that have ever simultaneously been in the
	 * pool.
	 */
	@ModelNodeBinding(detypedName = "largest-thread-count")
	public Integer largestThreadCount() {
		return this.largestThreadCount;
	}

	/**
	 * The largest number of threads that have ever simultaneously been in the
	 * pool.
	 */
	@SuppressWarnings("unchecked")
	public T largestThreadCount(Integer value) {
		Object oldValue = this.largestThreadCount;
		this.largestThreadCount = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("largestThreadCount", oldValue, value);
		return (T) this;
	}

	/**
	 * The maximum thread pool size.
	 */
	@ModelNodeBinding(detypedName = "max-threads")
	public Integer maxThreads() {
		return this.maxThreads;
	}

	/**
	 * The maximum thread pool size.
	 */
	@SuppressWarnings("unchecked")
	public T maxThreads(Integer value) {
		Object oldValue = this.maxThreads;
		this.maxThreads = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("maxThreads", oldValue, value);
		return (T) this;
	}

	/**
	 * The name of the thread pool.
	 */
	@ModelNodeBinding(detypedName = "name")
	public String name() {
		return this.name;
	}

	/**
	 * The name of the thread pool.
	 */
	@SuppressWarnings("unchecked")
	public T name(String value) {
		Object oldValue = this.name;
		this.name = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("name", oldValue, value);
		return (T) this;
	}

	/**
	 * The queue size.
	 */
	@ModelNodeBinding(detypedName = "queue-size")
	public Integer queueSize() {
		return this.queueSize;
	}

	/**
	 * The queue size.
	 */
	@SuppressWarnings("unchecked")
	public T queueSize(Integer value) {
		Object oldValue = this.queueSize;
		this.queueSize = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("queueSize", oldValue, value);
		return (T) this;
	}

	/**
	 * The number of tasks that have been rejected.
	 */
	@ModelNodeBinding(detypedName = "rejected-count")
	public Integer rejectedCount() {
		return this.rejectedCount;
	}

	/**
	 * The number of tasks that have been rejected.
	 */
	@SuppressWarnings("unchecked")
	public T rejectedCount(Integer value) {
		Object oldValue = this.rejectedCount;
		this.rejectedCount = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("rejectedCount", oldValue, value);
		return (T) this;
	}

	/**
	 * The approximate total number of tasks that have ever been scheduled for
	 * execution.
	 */
	@ModelNodeBinding(detypedName = "task-count")
	public Integer taskCount() {
		return this.taskCount;
	}

	/**
	 * The approximate total number of tasks that have ever been scheduled for
	 * execution.
	 */
	@SuppressWarnings("unchecked")
	public T taskCount(Integer value) {
		Object oldValue = this.taskCount;
		this.taskCount = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("taskCount", oldValue, value);
		return (T) this;
	}

	/**
	 * Specifies the name of a specific thread factory to use to create worker
	 * threads. If not defined an appropriate default thread factory will be
	 * used.
	 */
	@ModelNodeBinding(detypedName = "thread-factory")
	public String threadFactory() {
		return this.threadFactory;
	}

	/**
	 * Specifies the name of a specific thread factory to use to create worker
	 * threads. If not defined an appropriate default thread factory will be
	 * used.
	 */
	@SuppressWarnings("unchecked")
	public T threadFactory(String value) {
		Object oldValue = this.threadFactory;
		this.threadFactory = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("threadFactory", oldValue, value);
		return (T) this;
	}
}