package org.wildfly.swarm.config.ee;

import org.wildfly.swarm.config.runtime.Address;
import org.wildfly.swarm.config.runtime.ResourceType;
import java.beans.PropertyChangeSupport;
import java.beans.PropertyChangeListener;
import org.wildfly.swarm.config.runtime.ModelNodeBinding;
import java.util.Arrays;
import org.wildfly.swarm.config.ee.RejectPolicy;
/**
 * A managed executor service
 */
@Address("/subsystem=ee/managed-executor-service=*")
@ResourceType("managed-executor-service")
public class ManagedExecutorService<T extends ManagedExecutorService<T>> {

	private String key;
	private PropertyChangeSupport pcs;
	private String contextService;
	private Integer coreThreads;
	private Long hungTaskThreshold;
	private String jndiName;
	private Long keepaliveTime;
	private Boolean longRunningTasks;
	private Integer maxThreads;
	private Integer queueLength;
	private RejectPolicy rejectPolicy;
	private String threadFactory;

	public ManagedExecutorService(String key) {
		this.key = key;
	}

	public String getKey() {
		return this.key;
	}

	/**
	 * Adds a property change listener
	 */
	public void addPropertyChangeListener(PropertyChangeListener listener) {
		if (null == this.pcs)
			this.pcs = new PropertyChangeSupport(this);
		this.pcs.addPropertyChangeListener(listener);
	}

	/**
	 * Removes a property change listener
	 */
	public void removePropertyChangeListener(PropertyChangeListener listener) {
		if (this.pcs != null)
			this.pcs.removePropertyChangeListener(listener);
	}

	/**
	 * The name of the context service to be used by the executor.
	 */
	@ModelNodeBinding(detypedName = "context-service")
	public String contextService() {
		return this.contextService;
	}

	/**
	 * The name of the context service to be used by the executor.
	 */
	@SuppressWarnings("unchecked")
	public T contextService(String value) {
		Object oldValue = this.contextService;
		this.contextService = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("contextService", oldValue, value);
		return (T) this;
	}

	/**
	 * The minimum number of threads to be used by the executor. If left
	 * undefined the default core-size is calculated based on the number of
	 * processors. A value of zero is not advised and in some cases invalid. See
	 * the queue-length attribute for details on how this value is used to
	 * determine the queuing strategy.
	 */
	@ModelNodeBinding(detypedName = "core-threads")
	public Integer coreThreads() {
		return this.coreThreads;
	}

	/**
	 * The minimum number of threads to be used by the executor. If left
	 * undefined the default core-size is calculated based on the number of
	 * processors. A value of zero is not advised and in some cases invalid. See
	 * the queue-length attribute for details on how this value is used to
	 * determine the queuing strategy.
	 */
	@SuppressWarnings("unchecked")
	public T coreThreads(Integer value) {
		Object oldValue = this.coreThreads;
		this.coreThreads = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("coreThreads", oldValue, value);
		return (T) this;
	}

	/**
	 * The runtime, in milliseconds, for tasks to be considered hung by the
	 * managed executor service. If value is 0 tasks are never considered hung.
	 */
	@ModelNodeBinding(detypedName = "hung-task-threshold")
	public Long hungTaskThreshold() {
		return this.hungTaskThreshold;
	}

	/**
	 * The runtime, in milliseconds, for tasks to be considered hung by the
	 * managed executor service. If value is 0 tasks are never considered hung.
	 */
	@SuppressWarnings("unchecked")
	public T hungTaskThreshold(Long value) {
		Object oldValue = this.hungTaskThreshold;
		this.hungTaskThreshold = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("hungTaskThreshold", oldValue, value);
		return (T) this;
	}

	/**
	 * The JNDI Name to lookup the managed executor service.
	 */
	@ModelNodeBinding(detypedName = "jndi-name")
	public String jndiName() {
		return this.jndiName;
	}

	/**
	 * The JNDI Name to lookup the managed executor service.
	 */
	@SuppressWarnings("unchecked")
	public T jndiName(String value) {
		Object oldValue = this.jndiName;
		this.jndiName = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("jndiName", oldValue, value);
		return (T) this;
	}

	/**
	 * When the number of threads is greater than the core, this is the maximum
	 * time, in milliseconds, that excess idle threads will wait for new tasks
	 * before terminating.
	 */
	@ModelNodeBinding(detypedName = "keepalive-time")
	public Long keepaliveTime() {
		return this.keepaliveTime;
	}

	/**
	 * When the number of threads is greater than the core, this is the maximum
	 * time, in milliseconds, that excess idle threads will wait for new tasks
	 * before terminating.
	 */
	@SuppressWarnings("unchecked")
	public T keepaliveTime(Long value) {
		Object oldValue = this.keepaliveTime;
		this.keepaliveTime = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("keepaliveTime", oldValue, value);
		return (T) this;
	}

	/**
	 * Flag which hints the duration of tasks executed by the executor.
	 */
	@ModelNodeBinding(detypedName = "long-running-tasks")
	public Boolean longRunningTasks() {
		return this.longRunningTasks;
	}

	/**
	 * Flag which hints the duration of tasks executed by the executor.
	 */
	@SuppressWarnings("unchecked")
	public T longRunningTasks(Boolean value) {
		Object oldValue = this.longRunningTasks;
		this.longRunningTasks = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("longRunningTasks", oldValue, value);
		return (T) this;
	}

	/**
	 * The maximum number of threads to be used by the executor. If left
	 * undefined the value from core-size will be used. This value is ignored if
	 * an unbounded queue is used (only core-threads will be used in that case).
	 */
	@ModelNodeBinding(detypedName = "max-threads")
	public Integer maxThreads() {
		return this.maxThreads;
	}

	/**
	 * The maximum number of threads to be used by the executor. If left
	 * undefined the value from core-size will be used. This value is ignored if
	 * an unbounded queue is used (only core-threads will be used in that case).
	 */
	@SuppressWarnings("unchecked")
	public T maxThreads(Integer value) {
		Object oldValue = this.maxThreads;
		this.maxThreads = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("maxThreads", oldValue, value);
		return (T) this;
	}

	/**
	 * The executors task queue capacity. A length of 0 means direct hand-off
	 * and possible rejection will occur. An undefined length (the default), or
	 * Integer.MAX_VALUE, indicates that an unbounded queue should be used. All
	 * other values specify an exact queue size. If an unbounded queue or direct
	 * hand-off is used, a core-threads value greater than zero is required.
	 */
	@ModelNodeBinding(detypedName = "queue-length")
	public Integer queueLength() {
		return this.queueLength;
	}

	/**
	 * The executors task queue capacity. A length of 0 means direct hand-off
	 * and possible rejection will occur. An undefined length (the default), or
	 * Integer.MAX_VALUE, indicates that an unbounded queue should be used. All
	 * other values specify an exact queue size. If an unbounded queue or direct
	 * hand-off is used, a core-threads value greater than zero is required.
	 */
	@SuppressWarnings("unchecked")
	public T queueLength(Integer value) {
		Object oldValue = this.queueLength;
		this.queueLength = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("queueLength", oldValue, value);
		return (T) this;
	}

	/**
	 * The policy to be applied to aborted tasks.
	 */
	@ModelNodeBinding(detypedName = "reject-policy")
	public RejectPolicy rejectPolicy() {
		return this.rejectPolicy;
	}

	/**
	 * The policy to be applied to aborted tasks.
	 */
	@SuppressWarnings("unchecked")
	public T rejectPolicy(RejectPolicy value) {
		Object oldValue = this.rejectPolicy;
		this.rejectPolicy = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("rejectPolicy", oldValue, value);
		return (T) this;
	}

	/**
	 * The name of the thread factory to be used by the executor.
	 */
	@ModelNodeBinding(detypedName = "thread-factory")
	public String threadFactory() {
		return this.threadFactory;
	}

	/**
	 * The name of the thread factory to be used by the executor.
	 */
	@SuppressWarnings("unchecked")
	public T threadFactory(String value) {
		Object oldValue = this.threadFactory;
		this.threadFactory = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("threadFactory", oldValue, value);
		return (T) this;
	}
}