package org.wildfly.swarm.config.singleton;

import org.wildfly.swarm.config.runtime.Address;
import org.wildfly.swarm.config.runtime.ResourceType;
import java.beans.PropertyChangeSupport;
import java.beans.PropertyChangeListener;
import java.util.List;
import org.wildfly.swarm.config.runtime.Subresource;
import org.wildfly.swarm.config.singleton.singleton_policy.RandomElectionPolicy;
import org.wildfly.swarm.config.singleton.singleton_policy.RandomElectionPolicyConsumer;
import org.wildfly.swarm.config.singleton.singleton_policy.RandomElectionPolicySupplier;
import org.wildfly.swarm.config.singleton.singleton_policy.SimpleElectionPolicy;
import org.wildfly.swarm.config.singleton.singleton_policy.SimpleElectionPolicyConsumer;
import org.wildfly.swarm.config.singleton.singleton_policy.SimpleElectionPolicySupplier;
import org.wildfly.swarm.config.runtime.ModelNodeBinding;
/**
 * A singleton policy
 */
@Address("/subsystem=singleton/singleton-policy=*")
@ResourceType("singleton-policy")
public class SingletonPolicy<T extends SingletonPolicy<T>> {

	private String key;
	private PropertyChangeSupport pcs;
	private SingletonPolicyResources subresources = new SingletonPolicyResources();
	private String cache;
	private String cacheContainer;
	private Integer quorum;

	public SingletonPolicy(String key) {
		this.key = key;
	}

	public String getKey() {
		return this.key;
	}

	/**
	 * Adds a property change listener
	 */
	public void addPropertyChangeListener(PropertyChangeListener listener) {
		if (null == this.pcs)
			this.pcs = new PropertyChangeSupport(this);
		this.pcs.addPropertyChangeListener(listener);
	}

	/**
	 * Removes a property change listener
	 */
	public void removePropertyChangeListener(PropertyChangeListener listener) {
		if (this.pcs != null)
			this.pcs.removePropertyChangeListener(listener);
	}

	public SingletonPolicyResources subresources() {
		return this.subresources;
	}

	/**
	 * A random election policy
	 */
	@SuppressWarnings("unchecked")
	public T randomElectionPolicy(RandomElectionPolicy value) {
		this.subresources.randomElectionPolicy = value;
		return (T) this;
	}

	/**
	 * A random election policy
	 */
	@SuppressWarnings("unchecked")
	public T randomElectionPolicy(RandomElectionPolicyConsumer consumer) {
		RandomElectionPolicy<? extends RandomElectionPolicy> child = new RandomElectionPolicy<>();
		if (consumer != null) {
			consumer.accept(child);
		}
		this.subresources.randomElectionPolicy = child;
		return (T) this;
	}

	/**
	 * A random election policy
	 */
	@SuppressWarnings("unchecked")
	public T randomElectionPolicy() {
		RandomElectionPolicy<? extends RandomElectionPolicy> child = new RandomElectionPolicy<>();
		this.subresources.randomElectionPolicy = child;
		return (T) this;
	}

	/**
	 * A random election policy
	 */
	@SuppressWarnings("unchecked")
	public T randomElectionPolicy(RandomElectionPolicySupplier supplier) {
		this.subresources.randomElectionPolicy = supplier.get();
		return (T) this;
	}

	/**
	 * A simple election policy
	 */
	@SuppressWarnings("unchecked")
	public T simpleElectionPolicy(SimpleElectionPolicy value) {
		this.subresources.simpleElectionPolicy = value;
		return (T) this;
	}

	/**
	 * A simple election policy
	 */
	@SuppressWarnings("unchecked")
	public T simpleElectionPolicy(SimpleElectionPolicyConsumer consumer) {
		SimpleElectionPolicy<? extends SimpleElectionPolicy> child = new SimpleElectionPolicy<>();
		if (consumer != null) {
			consumer.accept(child);
		}
		this.subresources.simpleElectionPolicy = child;
		return (T) this;
	}

	/**
	 * A simple election policy
	 */
	@SuppressWarnings("unchecked")
	public T simpleElectionPolicy() {
		SimpleElectionPolicy<? extends SimpleElectionPolicy> child = new SimpleElectionPolicy<>();
		this.subresources.simpleElectionPolicy = child;
		return (T) this;
	}

	/**
	 * A simple election policy
	 */
	@SuppressWarnings("unchecked")
	public T simpleElectionPolicy(SimpleElectionPolicySupplier supplier) {
		this.subresources.simpleElectionPolicy = supplier.get();
		return (T) this;
	}

	/**
	 * Child mutators for SingletonPolicy
	 */
	public static class SingletonPolicyResources {
		private RandomElectionPolicy randomElectionPolicy;
		private SimpleElectionPolicy simpleElectionPolicy;

		/**
		 * A random election policy
		 */
		@Subresource
		public RandomElectionPolicy randomElectionPolicy() {
			return this.randomElectionPolicy;
		}

		/**
		 * A simple election policy
		 */
		@Subresource
		public SimpleElectionPolicy simpleElectionPolicy() {
			return this.simpleElectionPolicy;
		}
	}

	/**
	 * The cache backing the singleton policy's singleton service
	 */
	@ModelNodeBinding(detypedName = "cache")
	public String cache() {
		return this.cache;
	}

	/**
	 * The cache backing the singleton policy's singleton service
	 */
	@SuppressWarnings("unchecked")
	public T cache(String value) {
		Object oldValue = this.cache;
		this.cache = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("cache", oldValue, value);
		return (T) this;
	}

	/**
	 * The cache container backing the singleton policy's singleton service
	 */
	@ModelNodeBinding(detypedName = "cache-container")
	public String cacheContainer() {
		return this.cacheContainer;
	}

	/**
	 * The cache container backing the singleton policy's singleton service
	 */
	@SuppressWarnings("unchecked")
	public T cacheContainer(String value) {
		Object oldValue = this.cacheContainer;
		this.cacheContainer = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("cacheContainer", oldValue, value);
		return (T) this;
	}

	/**
	 * The minimum number of nodes required before this singleton service will
	 * start
	 */
	@ModelNodeBinding(detypedName = "quorum")
	public Integer quorum() {
		return this.quorum;
	}

	/**
	 * The minimum number of nodes required before this singleton service will
	 * start
	 */
	@SuppressWarnings("unchecked")
	public T quorum(Integer value) {
		Object oldValue = this.quorum;
		this.quorum = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("quorum", oldValue, value);
		return (T) this;
	}
}