package org.wildfly.swarm.config;

import org.wildfly.swarm.config.runtime.Address;
import org.wildfly.swarm.config.runtime.ResourceType;
import org.wildfly.swarm.config.runtime.Implicit;
import java.beans.PropertyChangeSupport;
import java.beans.PropertyChangeListener;
import java.util.List;
import org.wildfly.swarm.config.runtime.Subresource;
import org.wildfly.swarm.config.infinispan.CacheContainerConsumer;
import org.wildfly.swarm.config.infinispan.CacheContainerSupplier;
import org.wildfly.swarm.config.infinispan.CacheContainer;
import org.wildfly.swarm.config.runtime.ModelNodeBinding;
/**
 * The configuration of the infinispan subsystem.
 */
@Address("/subsystem=infinispan")
@ResourceType("subsystem")
@Implicit
public class Infinispan<T extends Infinispan<T>> {

	private String key;
	private PropertyChangeSupport pcs;
	private InfinispanResources subresources = new InfinispanResources();

	public Infinispan() {
		this.key = "infinispan";
		this.pcs = new PropertyChangeSupport(this);
	}

	public String getKey() {
		return this.key;
	}

	/**
	 * Adds a property change listener
	 */
	public void addPropertyChangeListener(PropertyChangeListener listener) {
		if (null == this.pcs)
			this.pcs = new PropertyChangeSupport(this);
		this.pcs.addPropertyChangeListener(listener);
	}

	/**
	 * Removes a property change listener
	 */
	public void removePropertyChangeListener(PropertyChangeListener listener) {
		if (this.pcs != null)
			this.pcs.removePropertyChangeListener(listener);
	}

	public InfinispanResources subresources() {
		return this.subresources;
	}

	/**
	 * Add all CacheContainer objects to this subresource
	 * 
	 * @return this
	 * @param value
	 *            List of CacheContainer objects.
	 */
	@SuppressWarnings("unchecked")
	public T cacheContainers(List<CacheContainer> value) {
		this.subresources.cacheContainers = value;
		return (T) this;
	}

	/**
	 * Add the CacheContainer object to the list of subresources
	 * 
	 * @param value
	 *            The CacheContainer to add
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T cacheContainer(CacheContainer value) {
		this.subresources.cacheContainers.add(value);
		return (T) this;
	}

	/**
	 * Create and configure a CacheContainer object to the list of subresources
	 * 
	 * @param key
	 *            The key for the CacheContainer resource
	 * @param config
	 *            The CacheContainerConsumer to use
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T cacheContainer(String childKey, CacheContainerConsumer consumer) {
		CacheContainer<? extends CacheContainer> child = new CacheContainer<>(
				childKey);
		if (consumer != null) {
			consumer.accept(child);
		}
		cacheContainer(child);
		return (T) this;
	}

	/**
	 * Create and configure a CacheContainer object to the list of subresources
	 * 
	 * @param key
	 *            The key for the CacheContainer resource
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T cacheContainer(String childKey) {
		cacheContainer(childKey, null);
		return (T) this;
	}

	/**
	 * Install a supplied CacheContainer object to the list of subresources
	 */
	@SuppressWarnings("unchecked")
	public T cacheContainer(CacheContainerSupplier supplier) {
		cacheContainer(supplier.get());
		return (T) this;
	}

	/**
	 * Child mutators for Infinispan
	 */
	public static class InfinispanResources {
		/**
		 * The configuration of an infinispan cache container
		 */
		private List<CacheContainer> cacheContainers = new java.util.ArrayList<>();

		/**
		 * Get the list of CacheContainer resources
		 * 
		 * @return the list of resources
		 */
		@Subresource
		public List<CacheContainer> cacheContainers() {
			return this.cacheContainers;
		}

		public CacheContainer cacheContainer(String key) {
			return this.cacheContainers.stream()
					.filter(e -> e.getKey().equals(key)).findFirst()
					.orElse(null);
		}
	}
}