package org.wildfly.swarm.config.ejb3;

import org.wildfly.swarm.config.runtime.Address;
import org.wildfly.swarm.config.runtime.ResourceType;
import java.beans.PropertyChangeSupport;
import java.beans.PropertyChangeListener;
import java.util.List;
import org.wildfly.swarm.config.runtime.Subresource;
import org.wildfly.swarm.config.ejb3.remoting_profile.RemotingEJBReceiverConsumer;
import org.wildfly.swarm.config.ejb3.remoting_profile.RemotingEJBReceiverSupplier;
import org.wildfly.swarm.config.ejb3.remoting_profile.RemotingEJBReceiver;
import org.wildfly.swarm.config.runtime.ModelNodeBinding;
/**
 * A remoting profile
 */
@Address("/subsystem=ejb3/remoting-profile=*")
@ResourceType("remoting-profile")
public class RemotingProfile<T extends RemotingProfile<T>> {

	private String key;
	private PropertyChangeSupport pcs;
	private RemotingProfileResources subresources = new RemotingProfileResources();
	private Boolean excludeLocalReceiver;
	private Boolean localReceiverPassByValue;

	public RemotingProfile(String key) {
		this.key = key;
	}

	public String getKey() {
		return this.key;
	}

	/**
	 * Adds a property change listener
	 */
	public void addPropertyChangeListener(PropertyChangeListener listener) {
		if (null == this.pcs)
			this.pcs = new PropertyChangeSupport(this);
		this.pcs.addPropertyChangeListener(listener);
	}

	/**
	 * Removes a property change listener
	 */
	public void removePropertyChangeListener(PropertyChangeListener listener) {
		if (this.pcs != null)
			this.pcs.removePropertyChangeListener(listener);
	}

	public RemotingProfileResources subresources() {
		return this.subresources;
	}

	/**
	 * Add all RemotingEJBReceiver objects to this subresource
	 * 
	 * @return this
	 * @param value
	 *            List of RemotingEJBReceiver objects.
	 */
	@SuppressWarnings("unchecked")
	public T remotingEjbReceivers(List<RemotingEJBReceiver> value) {
		this.subresources.remotingEjbReceivers = value;
		return (T) this;
	}

	/**
	 * Add the RemotingEJBReceiver object to the list of subresources
	 * 
	 * @param value
	 *            The RemotingEJBReceiver to add
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T remotingEjbReceiver(RemotingEJBReceiver value) {
		this.subresources.remotingEjbReceivers.add(value);
		return (T) this;
	}

	/**
	 * Create and configure a RemotingEJBReceiver object to the list of
	 * subresources
	 * 
	 * @param key
	 *            The key for the RemotingEJBReceiver resource
	 * @param config
	 *            The RemotingEJBReceiverConsumer to use
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T remotingEjbReceiver(String childKey,
			RemotingEJBReceiverConsumer consumer) {
		RemotingEJBReceiver<? extends RemotingEJBReceiver> child = new RemotingEJBReceiver<>(
				childKey);
		if (consumer != null) {
			consumer.accept(child);
		}
		remotingEjbReceiver(child);
		return (T) this;
	}

	/**
	 * Create and configure a RemotingEJBReceiver object to the list of
	 * subresources
	 * 
	 * @param key
	 *            The key for the RemotingEJBReceiver resource
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T remotingEjbReceiver(String childKey) {
		remotingEjbReceiver(childKey, null);
		return (T) this;
	}

	/**
	 * Install a supplied RemotingEJBReceiver object to the list of subresources
	 */
	@SuppressWarnings("unchecked")
	public T remotingEjbReceiver(RemotingEJBReceiverSupplier supplier) {
		remotingEjbReceiver(supplier.get());
		return (T) this;
	}

	/**
	 * Child mutators for RemotingProfile
	 */
	public static class RemotingProfileResources {
		/**
		 * A remoting ejb receiver reference included in the profile
		 */
		private List<RemotingEJBReceiver> remotingEjbReceivers = new java.util.ArrayList<>();

		/**
		 * Get the list of RemotingEJBReceiver resources
		 * 
		 * @return the list of resources
		 */
		@Subresource
		public List<RemotingEJBReceiver> remotingEjbReceivers() {
			return this.remotingEjbReceivers;
		}

		public RemotingEJBReceiver remotingEjbReceiver(String key) {
			return this.remotingEjbReceivers.stream()
					.filter(e -> e.getKey().equals(key)).findFirst()
					.orElse(null);
		}
	}

	/**
	 * If set no local receiver is used in this profile
	 */
	@ModelNodeBinding(detypedName = "exclude-local-receiver")
	public Boolean excludeLocalReceiver() {
		return this.excludeLocalReceiver;
	}

	/**
	 * If set no local receiver is used in this profile
	 */
	@SuppressWarnings("unchecked")
	public T excludeLocalReceiver(Boolean value) {
		Object oldValue = this.excludeLocalReceiver;
		this.excludeLocalReceiver = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("excludeLocalReceiver", oldValue, value);
		return (T) this;
	}

	/**
	 * If set local receiver will pass ejb beans by value
	 */
	@ModelNodeBinding(detypedName = "local-receiver-pass-by-value")
	public Boolean localReceiverPassByValue() {
		return this.localReceiverPassByValue;
	}

	/**
	 * If set local receiver will pass ejb beans by value
	 */
	@SuppressWarnings("unchecked")
	public T localReceiverPassByValue(Boolean value) {
		Object oldValue = this.localReceiverPassByValue;
		this.localReceiverPassByValue = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("localReceiverPassByValue", oldValue,
					value);
		return (T) this;
	}
}