package org.wildfly.swarm.config.ee;

import org.wildfly.swarm.config.runtime.Address;
import org.wildfly.swarm.config.runtime.ResourceType;
import java.beans.PropertyChangeSupport;
import java.beans.PropertyChangeListener;
import org.wildfly.swarm.config.runtime.ModelNodeBinding;
/**
 * A managed thread factory
 */
@Address("/subsystem=ee/managed-thread-factory=*")
@ResourceType("managed-thread-factory")
public class ManagedThreadFactory<T extends ManagedThreadFactory<T>> {

	private String key;
	private PropertyChangeSupport pcs;
	private String contextService;
	private String jndiName;
	private Integer priority;

	public ManagedThreadFactory(String key) {
		this.key = key;
	}

	public String getKey() {
		return this.key;
	}

	/**
	 * Adds a property change listener
	 */
	public void addPropertyChangeListener(PropertyChangeListener listener) {
		if (null == this.pcs)
			this.pcs = new PropertyChangeSupport(this);
		this.pcs.addPropertyChangeListener(listener);
	}

	/**
	 * Removes a property change listener
	 */
	public void removePropertyChangeListener(PropertyChangeListener listener) {
		if (this.pcs != null)
			this.pcs.removePropertyChangeListener(listener);
	}

	/**
	 * The name of the context service to be used by the managed thread factory
	 */
	@ModelNodeBinding(detypedName = "context-service")
	public String contextService() {
		return this.contextService;
	}

	/**
	 * The name of the context service to be used by the managed thread factory
	 */
	@SuppressWarnings("unchecked")
	public T contextService(String value) {
		Object oldValue = this.contextService;
		this.contextService = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("contextService", oldValue, value);
		return (T) this;
	}

	/**
	 * The JNDI Name to lookup the managed thread factory.
	 */
	@ModelNodeBinding(detypedName = "jndi-name")
	public String jndiName() {
		return this.jndiName;
	}

	/**
	 * The JNDI Name to lookup the managed thread factory.
	 */
	@SuppressWarnings("unchecked")
	public T jndiName(String value) {
		Object oldValue = this.jndiName;
		this.jndiName = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("jndiName", oldValue, value);
		return (T) this;
	}

	/**
	 * The priority applied to threads created by the factory
	 */
	@ModelNodeBinding(detypedName = "priority")
	public Integer priority() {
		return this.priority;
	}

	/**
	 * The priority applied to threads created by the factory
	 */
	@SuppressWarnings("unchecked")
	public T priority(Integer value) {
		Object oldValue = this.priority;
		this.priority = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("priority", oldValue, value);
		return (T) this;
	}
}