package org.wildfly.swarm.config.management.access.constraint.sensitivity_classification.type;

import org.wildfly.swarm.config.runtime.Address;
import org.wildfly.swarm.config.runtime.ResourceType;
import java.beans.PropertyChangeSupport;
import java.beans.PropertyChangeListener;
import java.util.List;
import org.wildfly.swarm.config.runtime.Subresource;
import org.wildfly.swarm.config.management.access.constraint.type.classification.AppliesToConsumer;
import org.wildfly.swarm.config.management.access.constraint.type.classification.AppliesToSupplier;
import org.wildfly.swarm.config.management.access.constraint.type.classification.AppliesTo;
import org.wildfly.swarm.config.runtime.ModelNodeBinding;
/**
 * Configuration of a sensitivity constraint.
 */
@Address("/core-service=management/access=authorization/constraint=sensitivity-classification/type=*/classification=*")
@ResourceType("classification")
public class Classification<T extends Classification<T>> {

	private String key;
	private PropertyChangeSupport pcs;
	private ClassificationResources subresources = new ClassificationResources();
	private Boolean configuredRequiresAddressable;
	private Boolean configuredRequiresRead;
	private Boolean configuredRequiresWrite;
	private Boolean defaultRequiresAddressable;
	private Boolean defaultRequiresRead;
	private Boolean defaultRequiresWrite;

	public Classification(String key) {
		this.key = key;
	}

	public String getKey() {
		return this.key;
	}

	/**
	 * Adds a property change listener
	 */
	public void addPropertyChangeListener(PropertyChangeListener listener) {
		if (null == this.pcs)
			this.pcs = new PropertyChangeSupport(this);
		this.pcs.addPropertyChangeListener(listener);
	}

	/**
	 * Removes a property change listener
	 */
	public void removePropertyChangeListener(PropertyChangeListener listener) {
		if (this.pcs != null)
			this.pcs.removePropertyChangeListener(listener);
	}

	public ClassificationResources subresources() {
		return this.subresources;
	}

	/**
	 * Add all AppliesTo objects to this subresource
	 * 
	 * @return this
	 * @param value
	 *            List of AppliesTo objects.
	 */
	@SuppressWarnings("unchecked")
	public T appliesTos(List<AppliesTo> value) {
		this.subresources.appliesTos = value;
		return (T) this;
	}

	/**
	 * Add the AppliesTo object to the list of subresources
	 * 
	 * @param value
	 *            The AppliesTo to add
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T appliesTo(AppliesTo value) {
		this.subresources.appliesTos.add(value);
		return (T) this;
	}

	/**
	 * Create and configure a AppliesTo object to the list of subresources
	 * 
	 * @param key
	 *            The key for the AppliesTo resource
	 * @param config
	 *            The AppliesToConsumer to use
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T appliesTo(String childKey, AppliesToConsumer consumer) {
		AppliesTo<? extends AppliesTo> child = new AppliesTo<>(childKey);
		if (consumer != null) {
			consumer.accept(child);
		}
		appliesTo(child);
		return (T) this;
	}

	/**
	 * Create and configure a AppliesTo object to the list of subresources
	 * 
	 * @param key
	 *            The key for the AppliesTo resource
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T appliesTo(String childKey) {
		appliesTo(childKey, null);
		return (T) this;
	}

	/**
	 * Install a supplied AppliesTo object to the list of subresources
	 */
	@SuppressWarnings("unchecked")
	public T appliesTo(AppliesToSupplier supplier) {
		appliesTo(supplier.get());
		return (T) this;
	}

	/**
	 * Child mutators for Classification
	 */
	public static class ClassificationResources {
		/**
		 * Information about the resources, attributes and operations to which
		 * an access control constraint applies.
		 */
		private List<AppliesTo> appliesTos = new java.util.ArrayList<>();

		/**
		 * Get the list of AppliesTo resources
		 * 
		 * @return the list of resources
		 */
		@Subresource
		public List<AppliesTo> appliesTos() {
			return this.appliesTos;
		}

		public AppliesTo appliesTo(String key) {
			return this.appliesTos.stream().filter(e -> e.getKey().equals(key))
					.findFirst().orElse(null);
		}
	}

	/**
	 * Set to override the default as to whether the visibility of resources
	 * annotated with this sensitivity constraint should be considered
	 * sensitive.
	 */
	@ModelNodeBinding(detypedName = "configured-requires-addressable")
	public Boolean configuredRequiresAddressable() {
		return this.configuredRequiresAddressable;
	}

	/**
	 * Set to override the default as to whether the visibility of resources
	 * annotated with this sensitivity constraint should be considered
	 * sensitive.
	 */
	@SuppressWarnings("unchecked")
	public T configuredRequiresAddressable(Boolean value) {
		Object oldValue = this.configuredRequiresAddressable;
		this.configuredRequiresAddressable = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("configuredRequiresAddressable",
					oldValue, value);
		return (T) this;
	}

	/**
	 * Set to override the default as to whether reading attributes annotated
	 * with this sensitivity constraint should be considered sensitive.
	 */
	@ModelNodeBinding(detypedName = "configured-requires-read")
	public Boolean configuredRequiresRead() {
		return this.configuredRequiresRead;
	}

	/**
	 * Set to override the default as to whether reading attributes annotated
	 * with this sensitivity constraint should be considered sensitive.
	 */
	@SuppressWarnings("unchecked")
	public T configuredRequiresRead(Boolean value) {
		Object oldValue = this.configuredRequiresRead;
		this.configuredRequiresRead = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("configuredRequiresRead", oldValue,
					value);
		return (T) this;
	}

	/**
	 * Set to override the default as to whether writing attributes annotated
	 * with this sensitivity constraint should be considered sensitive.
	 */
	@ModelNodeBinding(detypedName = "configured-requires-write")
	public Boolean configuredRequiresWrite() {
		return this.configuredRequiresWrite;
	}

	/**
	 * Set to override the default as to whether writing attributes annotated
	 * with this sensitivity constraint should be considered sensitive.
	 */
	@SuppressWarnings("unchecked")
	public T configuredRequiresWrite(Boolean value) {
		Object oldValue = this.configuredRequiresWrite;
		this.configuredRequiresWrite = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("configuredRequiresWrite", oldValue,
					value);
		return (T) this;
	}

	/**
	 * Whether the visibility of resources annotated with this sensitivity
	 * constraint should be considered sensitive.
	 */
	@ModelNodeBinding(detypedName = "default-requires-addressable")
	public Boolean defaultRequiresAddressable() {
		return this.defaultRequiresAddressable;
	}

	/**
	 * Whether the visibility of resources annotated with this sensitivity
	 * constraint should be considered sensitive.
	 */
	@SuppressWarnings("unchecked")
	public T defaultRequiresAddressable(Boolean value) {
		Object oldValue = this.defaultRequiresAddressable;
		this.defaultRequiresAddressable = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("defaultRequiresAddressable", oldValue,
					value);
		return (T) this;
	}

	/**
	 * Whether reading attributes annotated with this sensitivity constraint
	 * should be considered sensitive.
	 */
	@ModelNodeBinding(detypedName = "default-requires-read")
	public Boolean defaultRequiresRead() {
		return this.defaultRequiresRead;
	}

	/**
	 * Whether reading attributes annotated with this sensitivity constraint
	 * should be considered sensitive.
	 */
	@SuppressWarnings("unchecked")
	public T defaultRequiresRead(Boolean value) {
		Object oldValue = this.defaultRequiresRead;
		this.defaultRequiresRead = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("defaultRequiresRead", oldValue, value);
		return (T) this;
	}

	/**
	 * Whether writing attributes annotated with this sensitivity constraint
	 * should be considered sensitive.
	 */
	@ModelNodeBinding(detypedName = "default-requires-write")
	public Boolean defaultRequiresWrite() {
		return this.defaultRequiresWrite;
	}

	/**
	 * Whether writing attributes annotated with this sensitivity constraint
	 * should be considered sensitive.
	 */
	@SuppressWarnings("unchecked")
	public T defaultRequiresWrite(Boolean value) {
		Object oldValue = this.defaultRequiresWrite;
		this.defaultRequiresWrite = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("defaultRequiresWrite", oldValue, value);
		return (T) this;
	}
}