package org.wildfly.swarm.config.management.access.syslog_handler;

import org.wildfly.swarm.config.runtime.Address;
import org.wildfly.swarm.config.runtime.ResourceType;
import org.wildfly.swarm.config.runtime.Implicit;
import java.beans.PropertyChangeSupport;
import java.beans.PropertyChangeListener;
import java.util.List;
import org.wildfly.swarm.config.runtime.Subresource;
import org.wildfly.swarm.config.management.access.syslog_handler.protocol.TruststoreAuthentication;
import org.wildfly.swarm.config.management.access.syslog_handler.protocol.TruststoreAuthenticationConsumer;
import org.wildfly.swarm.config.management.access.syslog_handler.protocol.TruststoreAuthenticationSupplier;
import org.wildfly.swarm.config.management.access.syslog_handler.protocol.ClientCertificateStoreAuthentication;
import org.wildfly.swarm.config.management.access.syslog_handler.protocol.ClientCertificateStoreAuthenticationConsumer;
import org.wildfly.swarm.config.management.access.syslog_handler.protocol.ClientCertificateStoreAuthenticationSupplier;
import org.wildfly.swarm.config.runtime.ModelNodeBinding;
/**
 * Configuration to append to syslog over tls over tcp/ip.
 */
@Address("/core-service=management/access=audit/syslog-handler=*/protocol=tls")
@ResourceType("protocol")
@Implicit
public class TlsProtocol<T extends TlsProtocol<T>> {

	private String key;
	private PropertyChangeSupport pcs;
	private TlsProtocolResources subresources = new TlsProtocolResources();
	private String host;
	private String messageTransfer;
	private Integer port;
	private Integer reconnectTimeout;

	public TlsProtocol() {
		this.key = "tls";
		this.pcs = new PropertyChangeSupport(this);
	}

	public String getKey() {
		return this.key;
	}

	/**
	 * Adds a property change listener
	 */
	public void addPropertyChangeListener(PropertyChangeListener listener) {
		if (null == this.pcs)
			this.pcs = new PropertyChangeSupport(this);
		this.pcs.addPropertyChangeListener(listener);
	}

	/**
	 * Removes a property change listener
	 */
	public void removePropertyChangeListener(PropertyChangeListener listener) {
		if (this.pcs != null)
			this.pcs.removePropertyChangeListener(listener);
	}

	public TlsProtocolResources subresources() {
		return this.subresources;
	}

	/**
	 * Configuration for the truststore for the server certificate, if not
	 * signed by an authority.
	 */
	@SuppressWarnings("unchecked")
	public T truststoreAuthentication(TruststoreAuthentication value) {
		this.subresources.truststoreAuthentication = value;
		return (T) this;
	}

	/**
	 * Configuration for the truststore for the server certificate, if not
	 * signed by an authority.
	 */
	@SuppressWarnings("unchecked")
	public T truststoreAuthentication(TruststoreAuthenticationConsumer consumer) {
		TruststoreAuthentication<? extends TruststoreAuthentication> child = new TruststoreAuthentication<>();
		if (consumer != null) {
			consumer.accept(child);
		}
		this.subresources.truststoreAuthentication = child;
		return (T) this;
	}

	/**
	 * Configuration for the truststore for the server certificate, if not
	 * signed by an authority.
	 */
	@SuppressWarnings("unchecked")
	public T truststoreAuthentication() {
		TruststoreAuthentication<? extends TruststoreAuthentication> child = new TruststoreAuthentication<>();
		this.subresources.truststoreAuthentication = child;
		return (T) this;
	}

	/**
	 * Configuration for the truststore for the server certificate, if not
	 * signed by an authority.
	 */
	@SuppressWarnings("unchecked")
	public T truststoreAuthentication(TruststoreAuthenticationSupplier supplier) {
		this.subresources.truststoreAuthentication = supplier.get();
		return (T) this;
	}

	/**
	 * Configuration for the keystore containing the client certificate if the
	 * syslog server requires authentication.
	 */
	@SuppressWarnings("unchecked")
	public T clientCertificateStoreAuthentication(
			ClientCertificateStoreAuthentication value) {
		this.subresources.clientCertificateStoreAuthentication = value;
		return (T) this;
	}

	/**
	 * Configuration for the keystore containing the client certificate if the
	 * syslog server requires authentication.
	 */
	@SuppressWarnings("unchecked")
	public T clientCertificateStoreAuthentication(
			ClientCertificateStoreAuthenticationConsumer consumer) {
		ClientCertificateStoreAuthentication<? extends ClientCertificateStoreAuthentication> child = new ClientCertificateStoreAuthentication<>();
		if (consumer != null) {
			consumer.accept(child);
		}
		this.subresources.clientCertificateStoreAuthentication = child;
		return (T) this;
	}

	/**
	 * Configuration for the keystore containing the client certificate if the
	 * syslog server requires authentication.
	 */
	@SuppressWarnings("unchecked")
	public T clientCertificateStoreAuthentication() {
		ClientCertificateStoreAuthentication<? extends ClientCertificateStoreAuthentication> child = new ClientCertificateStoreAuthentication<>();
		this.subresources.clientCertificateStoreAuthentication = child;
		return (T) this;
	}

	/**
	 * Configuration for the keystore containing the client certificate if the
	 * syslog server requires authentication.
	 */
	@SuppressWarnings("unchecked")
	public T clientCertificateStoreAuthentication(
			ClientCertificateStoreAuthenticationSupplier supplier) {
		this.subresources.clientCertificateStoreAuthentication = supplier.get();
		return (T) this;
	}

	/**
	 * Child mutators for TlsProtocol
	 */
	public static class TlsProtocolResources {
		private TruststoreAuthentication truststoreAuthentication;
		private ClientCertificateStoreAuthentication clientCertificateStoreAuthentication;

		/**
		 * Configuration for the truststore for the server certificate, if not
		 * signed by an authority.
		 */
		@Subresource
		public TruststoreAuthentication truststoreAuthentication() {
			return this.truststoreAuthentication;
		}

		/**
		 * Configuration for the keystore containing the client certificate if
		 * the syslog server requires authentication.
		 */
		@Subresource
		public ClientCertificateStoreAuthentication clientCertificateStoreAuthentication() {
			return this.clientCertificateStoreAuthentication;
		}
	}

	/**
	 * The host of the syslog server for the tls over tcp requests.
	 */
	@ModelNodeBinding(detypedName = "host")
	public String host() {
		return this.host;
	}

	/**
	 * The host of the syslog server for the tls over tcp requests.
	 */
	@SuppressWarnings("unchecked")
	public T host(String value) {
		Object oldValue = this.host;
		this.host = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("host", oldValue, value);
		return (T) this;
	}

	/**
	 * The message transfer setting as described in section 3.4 of RFC-6587.
	 * This can either be OCTET_COUNTING as described in section 3.4.1 of
	 * RFC-6587, or NON_TRANSPARENT_FRAMING as described in section 3.4.1 of
	 * RFC-6587. See your syslog provider's documentation for what is supported.
	 */
	@ModelNodeBinding(detypedName = "message-transfer")
	public String messageTransfer() {
		return this.messageTransfer;
	}

	/**
	 * The message transfer setting as described in section 3.4 of RFC-6587.
	 * This can either be OCTET_COUNTING as described in section 3.4.1 of
	 * RFC-6587, or NON_TRANSPARENT_FRAMING as described in section 3.4.1 of
	 * RFC-6587. See your syslog provider's documentation for what is supported.
	 */
	@SuppressWarnings("unchecked")
	public T messageTransfer(String value) {
		Object oldValue = this.messageTransfer;
		this.messageTransfer = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("messageTransfer", oldValue, value);
		return (T) this;
	}

	/**
	 * The port of the syslog server for the tls over tcp requests.
	 */
	@ModelNodeBinding(detypedName = "port")
	public Integer port() {
		return this.port;
	}

	/**
	 * The port of the syslog server for the tls over tcp requests.
	 */
	@SuppressWarnings("unchecked")
	public T port(Integer value) {
		Object oldValue = this.port;
		this.port = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("port", oldValue, value);
		return (T) this;
	}

	/**
	 * If a connection drop is detected, the number of seconds to wait before
	 * reconnecting. A negative number means don't reconnect automatically.
	 */
	@ModelNodeBinding(detypedName = "reconnect-timeout")
	public Integer reconnectTimeout() {
		return this.reconnectTimeout;
	}

	/**
	 * If a connection drop is detected, the number of seconds to wait before
	 * reconnecting. A negative number means don't reconnect automatically.
	 */
	@SuppressWarnings("unchecked")
	public T reconnectTimeout(Integer value) {
		Object oldValue = this.reconnectTimeout;
		this.reconnectTimeout = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("reconnectTimeout", oldValue, value);
		return (T) this;
	}
}