package org.wildfly.swarm.config.messaging.activemq.server;

import org.wildfly.swarm.config.runtime.Address;
import org.wildfly.swarm.config.runtime.ResourceType;
import org.wildfly.swarm.config.runtime.Implicit;
import java.beans.PropertyChangeSupport;
import java.beans.PropertyChangeListener;
import org.wildfly.swarm.config.runtime.ModelNodeBinding;
import java.util.List;
import java.util.Arrays;
/**
 * A messaging resource that allows you to configure High Availability for the
 * ActiveMQ server (the value of ha-policy can be live-only, replication-master,
 * replication-slave, or replication-colocated).
 */
@Address("/subsystem=messaging-activemq/server=*/ha-policy=replication-slave")
@ResourceType("ha-policy")
@Implicit
public class ReplicationSlaveHAPolicy<T extends ReplicationSlaveHAPolicy<T>> {

	private String key;
	private PropertyChangeSupport pcs;
	private Boolean allowFailback;
	private String clusterName;
	private String groupName;
	private Long initialReplicationSyncTimeout;
	private Integer maxSavedReplicatedJournalSize;
	private Boolean restartBackup;
	private Boolean scaleDown;
	private String scaleDownClusterName;
	private List<String> scaleDownConnectors;
	private String scaleDownDiscoveryGroup;
	private String scaleDownGroupName;

	public ReplicationSlaveHAPolicy() {
		this.key = "replication-slave";
		this.pcs = new PropertyChangeSupport(this);
	}

	public String getKey() {
		return this.key;
	}

	/**
	 * Adds a property change listener
	 */
	public void addPropertyChangeListener(PropertyChangeListener listener) {
		if (null == this.pcs)
			this.pcs = new PropertyChangeSupport(this);
		this.pcs.addPropertyChangeListener(listener);
	}

	/**
	 * Removes a property change listener
	 */
	public void removePropertyChangeListener(PropertyChangeListener listener) {
		if (this.pcs != null)
			this.pcs.removePropertyChangeListener(listener);
	}

	/**
	 * Whether a server will automatically stop when a another places a request
	 * to take over its place. The use case is when a regular server stops and
	 * its backup takes over its duties, later the main server restarts and
	 * requests the server (the former backup) to stop operating.
	 */
	@ModelNodeBinding(detypedName = "allow-failback")
	public Boolean allowFailback() {
		return this.allowFailback;
	}

	/**
	 * Whether a server will automatically stop when a another places a request
	 * to take over its place. The use case is when a regular server stops and
	 * its backup takes over its duties, later the main server restarts and
	 * requests the server (the former backup) to stop operating.
	 */
	@SuppressWarnings("unchecked")
	public T allowFailback(Boolean value) {
		Object oldValue = this.allowFailback;
		this.allowFailback = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("allowFailback", oldValue, value);
		return (T) this;
	}

	/**
	 * Name of the cluster used for replication.
	 */
	@ModelNodeBinding(detypedName = "cluster-name")
	public String clusterName() {
		return this.clusterName;
	}

	/**
	 * Name of the cluster used for replication.
	 */
	@SuppressWarnings("unchecked")
	public T clusterName(String value) {
		Object oldValue = this.clusterName;
		this.clusterName = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("clusterName", oldValue, value);
		return (T) this;
	}

	/**
	 * If set, backup servers will only pair with live servers with matching
	 * group-name.
	 */
	@ModelNodeBinding(detypedName = "group-name")
	public String groupName() {
		return this.groupName;
	}

	/**
	 * If set, backup servers will only pair with live servers with matching
	 * group-name.
	 */
	@SuppressWarnings("unchecked")
	public T groupName(String value) {
		Object oldValue = this.groupName;
		this.groupName = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("groupName", oldValue, value);
		return (T) this;
	}

	/**
	 * How long to wait until the initiation replication is synchronized.
	 */
	@ModelNodeBinding(detypedName = "initial-replication-sync-timeout")
	public Long initialReplicationSyncTimeout() {
		return this.initialReplicationSyncTimeout;
	}

	/**
	 * How long to wait until the initiation replication is synchronized.
	 */
	@SuppressWarnings("unchecked")
	public T initialReplicationSyncTimeout(Long value) {
		Object oldValue = this.initialReplicationSyncTimeout;
		this.initialReplicationSyncTimeout = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("initialReplicationSyncTimeout",
					oldValue, value);
		return (T) this;
	}

	/**
	 * This specifies how many times a replicated backup server can restart
	 * after moving its files on start. Once there are this number of backup
	 * journal files the server will stop permanently after if fails back.
	 */
	@ModelNodeBinding(detypedName = "max-saved-replicated-journal-size")
	public Integer maxSavedReplicatedJournalSize() {
		return this.maxSavedReplicatedJournalSize;
	}

	/**
	 * This specifies how many times a replicated backup server can restart
	 * after moving its files on start. Once there are this number of backup
	 * journal files the server will stop permanently after if fails back.
	 */
	@SuppressWarnings("unchecked")
	public T maxSavedReplicatedJournalSize(Integer value) {
		Object oldValue = this.maxSavedReplicatedJournalSize;
		this.maxSavedReplicatedJournalSize = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("maxSavedReplicatedJournalSize",
					oldValue, value);
		return (T) this;
	}

	/**
	 * Will this server, if a backup, restart once it has been stopped because
	 * of failback or scaling down.
	 */
	@ModelNodeBinding(detypedName = "restart-backup")
	public Boolean restartBackup() {
		return this.restartBackup;
	}

	/**
	 * Will this server, if a backup, restart once it has been stopped because
	 * of failback or scaling down.
	 */
	@SuppressWarnings("unchecked")
	public T restartBackup(Boolean value) {
		Object oldValue = this.restartBackup;
		this.restartBackup = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("restartBackup", oldValue, value);
		return (T) this;
	}

	/**
	 * Configure whether this server send its messages to another live server in
	 * the scale-down cluster when it is shutdown cleanly.
	 */
	@ModelNodeBinding(detypedName = "scale-down")
	public Boolean scaleDown() {
		return this.scaleDown;
	}

	/**
	 * Configure whether this server send its messages to another live server in
	 * the scale-down cluster when it is shutdown cleanly.
	 */
	@SuppressWarnings("unchecked")
	public T scaleDown(Boolean value) {
		Object oldValue = this.scaleDown;
		this.scaleDown = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("scaleDown", oldValue, value);
		return (T) this;
	}

	/**
	 * Name of the cluster used to scale down.
	 */
	@ModelNodeBinding(detypedName = "scale-down-cluster-name")
	public String scaleDownClusterName() {
		return this.scaleDownClusterName;
	}

	/**
	 * Name of the cluster used to scale down.
	 */
	@SuppressWarnings("unchecked")
	public T scaleDownClusterName(String value) {
		Object oldValue = this.scaleDownClusterName;
		this.scaleDownClusterName = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("scaleDownClusterName", oldValue, value);
		return (T) this;
	}

	/**
	 * List of connectors used to form the scale-down cluster.
	 */
	@ModelNodeBinding(detypedName = "scale-down-connectors")
	public List<String> scaleDownConnectors() {
		return this.scaleDownConnectors;
	}

	/**
	 * List of connectors used to form the scale-down cluster.
	 */
	@SuppressWarnings("unchecked")
	public T scaleDownConnectors(List<String> value) {
		Object oldValue = this.scaleDownConnectors;
		this.scaleDownConnectors = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("scaleDownConnectors", oldValue, value);
		return (T) this;
	}

	/**
	 * List of connectors used to form the scale-down cluster.
	 */
	@SuppressWarnings("unchecked")
	public T scaleDownConnector(String value) {
		if (this.scaleDownConnectors == null) {
			this.scaleDownConnectors = new java.util.ArrayList<>();
		}
		this.scaleDownConnectors.add(value);
		return (T) this;
	}

	/**
	 * List of connectors used to form the scale-down cluster.
	 */
	@SuppressWarnings("unchecked")
	public T scaleDownConnectors(String... args) {
		scaleDownConnectors(Arrays.asList(args));
		return (T) this;
	}

	/**
	 * Name of the discovery group used to build the scale-down cluster
	 */
	@ModelNodeBinding(detypedName = "scale-down-discovery-group")
	public String scaleDownDiscoveryGroup() {
		return this.scaleDownDiscoveryGroup;
	}

	/**
	 * Name of the discovery group used to build the scale-down cluster
	 */
	@SuppressWarnings("unchecked")
	public T scaleDownDiscoveryGroup(String value) {
		Object oldValue = this.scaleDownDiscoveryGroup;
		this.scaleDownDiscoveryGroup = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("scaleDownDiscoveryGroup", oldValue,
					value);
		return (T) this;
	}

	/**
	 * Name of the group used to scale down.
	 */
	@ModelNodeBinding(detypedName = "scale-down-group-name")
	public String scaleDownGroupName() {
		return this.scaleDownGroupName;
	}

	/**
	 * Name of the group used to scale down.
	 */
	@SuppressWarnings("unchecked")
	public T scaleDownGroupName(String value) {
		Object oldValue = this.scaleDownGroupName;
		this.scaleDownGroupName = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("scaleDownGroupName", oldValue, value);
		return (T) this;
	}
}