package org.wildfly.swarm.config.security.security_domain;

import org.wildfly.swarm.config.runtime.Address;
import org.wildfly.swarm.config.runtime.ResourceType;
import org.wildfly.swarm.config.runtime.Implicit;
import java.beans.PropertyChangeSupport;
import java.beans.PropertyChangeListener;
import java.util.List;
import org.wildfly.swarm.config.runtime.Subresource;
import org.wildfly.swarm.config.security.security_domain.mapping.MappingModuleConsumer;
import org.wildfly.swarm.config.security.security_domain.mapping.MappingModuleSupplier;
import org.wildfly.swarm.config.security.security_domain.mapping.MappingModule;
import org.wildfly.swarm.config.runtime.ModelNodeBinding;
/**
 * Mapping configuration. Configures a list of mapping modules to be used for
 * principal, role, attribute and credential mapping.
 */
@Address("/subsystem=security/security-domain=*/mapping=classic")
@ResourceType("mapping")
@Implicit
public class ClassicMapping<T extends ClassicMapping<T>> {

	private String key;
	private PropertyChangeSupport pcs;
	private ClassicMappingResources subresources = new ClassicMappingResources();

	public ClassicMapping() {
		this.key = "classic";
		this.pcs = new PropertyChangeSupport(this);
	}

	public String getKey() {
		return this.key;
	}

	/**
	 * Adds a property change listener
	 */
	public void addPropertyChangeListener(PropertyChangeListener listener) {
		if (null == this.pcs)
			this.pcs = new PropertyChangeSupport(this);
		this.pcs.addPropertyChangeListener(listener);
	}

	/**
	 * Removes a property change listener
	 */
	public void removePropertyChangeListener(PropertyChangeListener listener) {
		if (this.pcs != null)
			this.pcs.removePropertyChangeListener(listener);
	}

	public ClassicMappingResources subresources() {
		return this.subresources;
	}

	/**
	 * Add all MappingModule objects to this subresource
	 * 
	 * @return this
	 * @param value
	 *            List of MappingModule objects.
	 */
	@SuppressWarnings("unchecked")
	public T mappingModules(List<MappingModule> value) {
		this.subresources.mappingModules = value;
		return (T) this;
	}

	/**
	 * Add the MappingModule object to the list of subresources
	 * 
	 * @param value
	 *            The MappingModule to add
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T mappingModule(MappingModule value) {
		this.subresources.mappingModules.add(value);
		return (T) this;
	}

	/**
	 * Create and configure a MappingModule object to the list of subresources
	 * 
	 * @param key
	 *            The key for the MappingModule resource
	 * @param config
	 *            The MappingModuleConsumer to use
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T mappingModule(String childKey, MappingModuleConsumer consumer) {
		MappingModule<? extends MappingModule> child = new MappingModule<>(
				childKey);
		if (consumer != null) {
			consumer.accept(child);
		}
		mappingModule(child);
		return (T) this;
	}

	/**
	 * Create and configure a MappingModule object to the list of subresources
	 * 
	 * @param key
	 *            The key for the MappingModule resource
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T mappingModule(String childKey) {
		mappingModule(childKey, null);
		return (T) this;
	}

	/**
	 * Install a supplied MappingModule object to the list of subresources
	 */
	@SuppressWarnings("unchecked")
	public T mappingModule(MappingModuleSupplier supplier) {
		mappingModule(supplier.get());
		return (T) this;
	}

	/**
	 * Child mutators for ClassicMapping
	 */
	public static class ClassicMappingResources {
		/**
		 * List of modules that map principal, role, and credential information
		 */
		private List<MappingModule> mappingModules = new java.util.ArrayList<>();

		/**
		 * Get the list of MappingModule resources
		 * 
		 * @return the list of resources
		 */
		@Subresource
		public List<MappingModule> mappingModules() {
			return this.mappingModules;
		}

		public MappingModule mappingModule(String key) {
			return this.mappingModules.stream()
					.filter(e -> e.getKey().equals(key)).findFirst()
					.orElse(null);
		}
	}
}