package org.wildfly.swarm.config.jmx;

import org.wildfly.swarm.config.runtime.Address;
import java.util.HashMap;
import org.wildfly.swarm.config.runtime.ResourceType;
import org.wildfly.swarm.config.runtime.Implicit;
import java.beans.PropertyChangeSupport;
import java.beans.PropertyChangeListener;
import org.wildfly.swarm.config.runtime.ModelNodeBinding;
/**
 * The configuration for exposing the 'resolved' model controller in the
 * MBeanServer. This facade will expose all simple attributes and operation
 * parameters as their type in the underlying model. Reads return the resolved
 * expression if used, or the raw value. You may not use expressions when
 * writing attributes and setting operation parameters.
 */
@Address("/subsystem=jmx/expose-model=resolved")
@ResourceType("expose-model")
@Implicit
public class ResolvedExposeModel<T extends ResolvedExposeModel<T>>
		extends
			HashMap {

	private String key;
	private PropertyChangeSupport pcs;
	private String domainName;
	private Boolean properPropertyFormat;

	public ResolvedExposeModel() {
		super();
		this.key = "resolved";
		this.pcs = new PropertyChangeSupport(this);
	}

	public String getKey() {
		return this.key;
	}

	/**
	 * Adds a property change listener
	 */
	public void addPropertyChangeListener(PropertyChangeListener listener) {
		if (null == this.pcs)
			this.pcs = new PropertyChangeSupport(this);
		this.pcs.addPropertyChangeListener(listener);
	}

	/**
	 * Removes a property change listener
	 */
	public void removePropertyChangeListener(PropertyChangeListener listener) {
		if (this.pcs != null)
			this.pcs.removePropertyChangeListener(listener);
	}

	/**
	 * The domain name to use for the 'resolved' model controller JMX facade in
	 * the MBeanServer.
	 */
	@ModelNodeBinding(detypedName = "domain-name")
	public String domainName() {
		return this.domainName;
	}

	/**
	 * The domain name to use for the 'resolved' model controller JMX facade in
	 * the MBeanServer.
	 */
	@SuppressWarnings("unchecked")
	public T domainName(String value) {
		Object oldValue = this.domainName;
		this.domainName = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("domainName", oldValue, value);
		return (T) this;
	}

	/**
	 * If false, PROPERTY type attributes are represented as a DMR string, this
	 * is the legacy behaviour. If true, PROPERTY type attributes are
	 * represented by a composite type where the key is a string, and the value
	 * has the same type as the property in the underlying model.
	 */
	@ModelNodeBinding(detypedName = "proper-property-format")
	public Boolean properPropertyFormat() {
		return this.properPropertyFormat;
	}

	/**
	 * If false, PROPERTY type attributes are represented as a DMR string, this
	 * is the legacy behaviour. If true, PROPERTY type attributes are
	 * represented by a composite type where the key is a string, and the value
	 * has the same type as the property in the underlying model.
	 */
	@SuppressWarnings("unchecked")
	public T properPropertyFormat(Boolean value) {
		Object oldValue = this.properPropertyFormat;
		this.properPropertyFormat = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("properPropertyFormat", oldValue, value);
		return (T) this;
	}
}