package org.wildfly.swarm.config.logging;

import org.wildfly.swarm.config.runtime.Addresses;
import java.util.HashMap;
import org.wildfly.swarm.config.runtime.ResourceType;
import java.beans.PropertyChangeSupport;
import java.beans.PropertyChangeListener;
import org.wildfly.swarm.config.runtime.ModelNodeBinding;
import java.util.Arrays;
import org.wildfly.swarm.config.logging.Level;
import java.util.List;
/**
 * Defines a handler which writes to the sub-handlers in an asynchronous thread.
 * Used for handlers which introduce a substantial amount of lag.
 */
@Addresses({"/subsystem=logging/async-handler=*",
		"/subsystem=logging/logging-profile=*/async-handler=*"})
@ResourceType("async-handler")
public class AsyncHandler<T extends AsyncHandler<T>> extends HashMap {

	private String key;
	private PropertyChangeSupport pcs;
	private Boolean enabled;
	private String filterSpec;
	private Level level;
	private OverflowAction overflowAction;
	private Integer queueLength;
	private List<String> subhandlers;

	public AsyncHandler(String key) {
		super();
		this.key = key;
	}

	public String getKey() {
		return this.key;
	}

	/**
	 * Adds a property change listener
	 */
	public void addPropertyChangeListener(PropertyChangeListener listener) {
		if (null == this.pcs)
			this.pcs = new PropertyChangeSupport(this);
		this.pcs.addPropertyChangeListener(listener);
	}

	/**
	 * Removes a property change listener
	 */
	public void removePropertyChangeListener(PropertyChangeListener listener) {
		if (this.pcs != null)
			this.pcs.removePropertyChangeListener(listener);
	}

	public static enum OverflowAction {
		BLOCK("BLOCK"), DISCARD("DISCARD");
		private final String allowedValue;

		/**
		 * Returns the allowed value for the management model.
		 * 
		 * @return the allowed model value
		 */
		public String getAllowedValue() {
			return allowedValue;
		}

		OverflowAction(String allowedValue) {
			this.allowedValue = allowedValue;
		}

		@Override
		public String toString() {
			return allowedValue;
		}
	}

	/**
	 * If set to true the handler is enabled and functioning as normal, if set
	 * to false the handler is ignored when processing log messages.
	 */
	@ModelNodeBinding(detypedName = "enabled")
	public Boolean enabled() {
		return this.enabled;
	}

	/**
	 * If set to true the handler is enabled and functioning as normal, if set
	 * to false the handler is ignored when processing log messages.
	 */
	@SuppressWarnings("unchecked")
	public T enabled(Boolean value) {
		Object oldValue = this.enabled;
		this.enabled = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("enabled", oldValue, value);
		return (T) this;
	}

	/**
	 * A filter expression value to define a filter. Example for a filter that
	 * does not match a pattern: not(match("JBAS.*"))
	 */
	@ModelNodeBinding(detypedName = "filter-spec")
	public String filterSpec() {
		return this.filterSpec;
	}

	/**
	 * A filter expression value to define a filter. Example for a filter that
	 * does not match a pattern: not(match("JBAS.*"))
	 */
	@SuppressWarnings("unchecked")
	public T filterSpec(String value) {
		Object oldValue = this.filterSpec;
		this.filterSpec = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("filterSpec", oldValue, value);
		return (T) this;
	}

	/**
	 * The log level specifying which message levels will be logged by this
	 * handler. Message levels lower than this value will be discarded.
	 */
	@ModelNodeBinding(detypedName = "level")
	public Level level() {
		return this.level;
	}

	/**
	 * The log level specifying which message levels will be logged by this
	 * handler. Message levels lower than this value will be discarded.
	 */
	@SuppressWarnings("unchecked")
	public T level(Level value) {
		Object oldValue = this.level;
		this.level = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("level", oldValue, value);
		return (T) this;
	}

	/**
	 * Specify what action to take when the overflowing. The valid options are
	 * 'block' and 'discard'
	 */
	@ModelNodeBinding(detypedName = "overflow-action")
	public OverflowAction overflowAction() {
		return this.overflowAction;
	}

	/**
	 * Specify what action to take when the overflowing. The valid options are
	 * 'block' and 'discard'
	 */
	@SuppressWarnings("unchecked")
	public T overflowAction(OverflowAction value) {
		Object oldValue = this.overflowAction;
		this.overflowAction = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("overflowAction", oldValue, value);
		return (T) this;
	}

	/**
	 * The queue length to use before flushing writing
	 */
	@ModelNodeBinding(detypedName = "queue-length")
	public Integer queueLength() {
		return this.queueLength;
	}

	/**
	 * The queue length to use before flushing writing
	 */
	@SuppressWarnings("unchecked")
	public T queueLength(Integer value) {
		Object oldValue = this.queueLength;
		this.queueLength = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("queueLength", oldValue, value);
		return (T) this;
	}

	/**
	 * The Handlers associated with this async handler.
	 */
	@ModelNodeBinding(detypedName = "subhandlers")
	public List<String> subhandlers() {
		return this.subhandlers;
	}

	/**
	 * The Handlers associated with this async handler.
	 */
	@SuppressWarnings("unchecked")
	public T subhandlers(List<String> value) {
		Object oldValue = this.subhandlers;
		this.subhandlers = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("subhandlers", oldValue, value);
		return (T) this;
	}

	/**
	 * The Handlers associated with this async handler.
	 */
	@SuppressWarnings("unchecked")
	public T subhandler(String value) {
		if (this.subhandlers == null) {
			this.subhandlers = new java.util.ArrayList<>();
		}
		this.subhandlers.add(value);
		return (T) this;
	}

	/**
	 * The Handlers associated with this async handler.
	 */
	@SuppressWarnings("unchecked")
	public T subhandlers(String... args) {
		subhandlers(Arrays.asList(args));
		return (T) this;
	}
}