package org.wildfly.swarm.config.logging;

import org.wildfly.swarm.config.runtime.Addresses;
import java.util.HashMap;
import org.wildfly.swarm.config.runtime.ResourceType;
import org.wildfly.swarm.config.runtime.Implicit;
import java.beans.PropertyChangeSupport;
import java.beans.PropertyChangeListener;
import org.wildfly.swarm.config.runtime.ModelNodeBinding;
import java.util.List;
import java.util.Arrays;
import org.wildfly.swarm.config.logging.Level;
/**
 * Defines the root logger for this log context.
 */
@Addresses({"/subsystem=logging/logging-profile=*/root-logger=ROOT",
		"/subsystem=logging/root-logger=ROOT"})
@ResourceType("root-logger")
@Implicit
public class RootLogger<T extends RootLogger<T>> extends HashMap {

	private String key;
	private PropertyChangeSupport pcs;
	private String filterSpec;
	private List<String> handlers;
	private Level level;

	public RootLogger() {
		super();
		this.key = "ROOT";
		this.pcs = new PropertyChangeSupport(this);
	}

	public String getKey() {
		return this.key;
	}

	/**
	 * Adds a property change listener
	 */
	public void addPropertyChangeListener(PropertyChangeListener listener) {
		if (null == this.pcs)
			this.pcs = new PropertyChangeSupport(this);
		this.pcs.addPropertyChangeListener(listener);
	}

	/**
	 * Removes a property change listener
	 */
	public void removePropertyChangeListener(PropertyChangeListener listener) {
		if (this.pcs != null)
			this.pcs.removePropertyChangeListener(listener);
	}

	/**
	 * A filter expression value to define a filter. Example for a filter that
	 * does not match a pattern: not(match("JBAS.*"))
	 */
	@ModelNodeBinding(detypedName = "filter-spec")
	public String filterSpec() {
		return this.filterSpec;
	}

	/**
	 * A filter expression value to define a filter. Example for a filter that
	 * does not match a pattern: not(match("JBAS.*"))
	 */
	@SuppressWarnings("unchecked")
	public T filterSpec(String value) {
		Object oldValue = this.filterSpec;
		this.filterSpec = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("filterSpec", oldValue, value);
		return (T) this;
	}

	/**
	 * The handlers associated with the root logger.
	 */
	@ModelNodeBinding(detypedName = "handlers")
	public List<String> handlers() {
		return this.handlers;
	}

	/**
	 * The handlers associated with the root logger.
	 */
	@SuppressWarnings("unchecked")
	public T handlers(List<String> value) {
		Object oldValue = this.handlers;
		this.handlers = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("handlers", oldValue, value);
		return (T) this;
	}

	/**
	 * The handlers associated with the root logger.
	 */
	@SuppressWarnings("unchecked")
	public T handler(String value) {
		if (this.handlers == null) {
			this.handlers = new java.util.ArrayList<>();
		}
		this.handlers.add(value);
		return (T) this;
	}

	/**
	 * The handlers associated with the root logger.
	 */
	@SuppressWarnings("unchecked")
	public T handlers(String... args) {
		handlers(Arrays.asList(args));
		return (T) this;
	}

	/**
	 * The log level specifying which message levels will be logged by the root
	 * logger. Message levels lower than this value will be discarded.
	 */
	@ModelNodeBinding(detypedName = "level")
	public Level level() {
		return this.level;
	}

	/**
	 * The log level specifying which message levels will be logged by the root
	 * logger. Message levels lower than this value will be discarded.
	 */
	@SuppressWarnings("unchecked")
	public T level(Level value) {
		Object oldValue = this.level;
		this.level = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("level", oldValue, value);
		return (T) this;
	}
}