package org.wildfly.swarm.config.management.access;

import org.wildfly.swarm.config.runtime.Address;
import java.util.HashMap;
import org.wildfly.swarm.config.runtime.ResourceType;
import java.beans.PropertyChangeSupport;
import java.beans.PropertyChangeListener;
import java.util.List;
import org.wildfly.swarm.config.runtime.Subresource;
import org.wildfly.swarm.config.management.access.role_mapping.IncludeConsumer;
import org.wildfly.swarm.config.management.access.role_mapping.IncludeSupplier;
import org.wildfly.swarm.config.management.access.role_mapping.Include;
import org.wildfly.swarm.config.management.access.role_mapping.ExcludeConsumer;
import org.wildfly.swarm.config.management.access.role_mapping.ExcludeSupplier;
import org.wildfly.swarm.config.management.access.role_mapping.Exclude;
import org.wildfly.swarm.config.runtime.ModelNodeBinding;
/**
 * A mapping of users and groups to a specific role.
 */
@Address("/core-service=management/access=authorization/role-mapping=*")
@ResourceType("role-mapping")
public class RoleMapping<T extends RoleMapping<T>> extends HashMap {

	private String key;
	private PropertyChangeSupport pcs;
	private RoleMappingResources subresources = new RoleMappingResources();
	private Boolean includeAll;

	public RoleMapping(String key) {
		super();
		this.key = key;
	}

	public String getKey() {
		return this.key;
	}

	/**
	 * Adds a property change listener
	 */
	public void addPropertyChangeListener(PropertyChangeListener listener) {
		if (null == this.pcs)
			this.pcs = new PropertyChangeSupport(this);
		this.pcs.addPropertyChangeListener(listener);
	}

	/**
	 * Removes a property change listener
	 */
	public void removePropertyChangeListener(PropertyChangeListener listener) {
		if (this.pcs != null)
			this.pcs.removePropertyChangeListener(listener);
	}

	public RoleMappingResources subresources() {
		return this.subresources;
	}

	/**
	 * Add all Include objects to this subresource
	 * 
	 * @return this
	 * @param value
	 *            List of Include objects.
	 */
	@SuppressWarnings("unchecked")
	public T includes(List<Include> value) {
		this.subresources.includes = value;
		return (T) this;
	}

	/**
	 * Add the Include object to the list of subresources
	 * 
	 * @param value
	 *            The Include to add
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T include(Include value) {
		this.subresources.includes.add(value);
		return (T) this;
	}

	/**
	 * Create and configure a Include object to the list of subresources
	 * 
	 * @param key
	 *            The key for the Include resource
	 * @param config
	 *            The IncludeConsumer to use
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T include(String childKey, IncludeConsumer consumer) {
		Include<? extends Include> child = new Include<>(childKey);
		if (consumer != null) {
			consumer.accept(child);
		}
		include(child);
		return (T) this;
	}

	/**
	 * Create and configure a Include object to the list of subresources
	 * 
	 * @param key
	 *            The key for the Include resource
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T include(String childKey) {
		include(childKey, null);
		return (T) this;
	}

	/**
	 * Install a supplied Include object to the list of subresources
	 */
	@SuppressWarnings("unchecked")
	public T include(IncludeSupplier supplier) {
		include(supplier.get());
		return (T) this;
	}

	/**
	 * Add all Exclude objects to this subresource
	 * 
	 * @return this
	 * @param value
	 *            List of Exclude objects.
	 */
	@SuppressWarnings("unchecked")
	public T excludes(List<Exclude> value) {
		this.subresources.excludes = value;
		return (T) this;
	}

	/**
	 * Add the Exclude object to the list of subresources
	 * 
	 * @param value
	 *            The Exclude to add
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T exclude(Exclude value) {
		this.subresources.excludes.add(value);
		return (T) this;
	}

	/**
	 * Create and configure a Exclude object to the list of subresources
	 * 
	 * @param key
	 *            The key for the Exclude resource
	 * @param config
	 *            The ExcludeConsumer to use
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T exclude(String childKey, ExcludeConsumer consumer) {
		Exclude<? extends Exclude> child = new Exclude<>(childKey);
		if (consumer != null) {
			consumer.accept(child);
		}
		exclude(child);
		return (T) this;
	}

	/**
	 * Create and configure a Exclude object to the list of subresources
	 * 
	 * @param key
	 *            The key for the Exclude resource
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T exclude(String childKey) {
		exclude(childKey, null);
		return (T) this;
	}

	/**
	 * Install a supplied Exclude object to the list of subresources
	 */
	@SuppressWarnings("unchecked")
	public T exclude(ExcludeSupplier supplier) {
		exclude(supplier.get());
		return (T) this;
	}

	/**
	 * Child mutators for RoleMapping
	 */
	public static class RoleMappingResources {
		/**
		 * An individual principal used within a role mapping.
		 */
		private List<Include> includes = new java.util.ArrayList<>();
		/**
		 * An individual principal used within a role mapping.
		 */
		private List<Exclude> excludes = new java.util.ArrayList<>();

		/**
		 * Get the list of Include resources
		 * 
		 * @return the list of resources
		 */
		@Subresource
		public List<Include> includes() {
			return this.includes;
		}

		public Include include(String key) {
			return this.includes.stream().filter(e -> e.getKey().equals(key))
					.findFirst().orElse(null);
		}
		/**
		 * Get the list of Exclude resources
		 * 
		 * @return the list of resources
		 */
		@Subresource
		public List<Exclude> excludes() {
			return this.excludes;
		}

		public Exclude exclude(String key) {
			return this.excludes.stream().filter(e -> e.getKey().equals(key))
					.findFirst().orElse(null);
		}
	}

	/**
	 * Configure if all authenticated users should be automatically assigned
	 * this role.
	 */
	@ModelNodeBinding(detypedName = "include-all")
	public Boolean includeAll() {
		return this.includeAll;
	}

	/**
	 * Configure if all authenticated users should be automatically assigned
	 * this role.
	 */
	@SuppressWarnings("unchecked")
	public T includeAll(Boolean value) {
		Object oldValue = this.includeAll;
		this.includeAll = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("includeAll", oldValue, value);
		return (T) this;
	}
}