package org.wildfly.swarm.config.management.access.constraint;

import org.wildfly.swarm.config.runtime.Addresses;
import java.util.HashMap;
import org.wildfly.swarm.config.runtime.ResourceType;
import java.beans.PropertyChangeSupport;
import java.beans.PropertyChangeListener;
import java.util.List;
import org.wildfly.swarm.config.runtime.Subresource;
import org.wildfly.swarm.config.management.access.constraint.application_classification.type.ClassificationConsumer;
import org.wildfly.swarm.config.management.access.constraint.application_classification.type.ClassificationSupplier;
import org.wildfly.swarm.config.management.access.constraint.application_classification.type.Classification;
import org.wildfly.swarm.config.runtime.ModelNodeBinding;
/**
 * The application classification constraints by type. Type is either 'core' or
 * the name of a subsystem.
 */
@Addresses({
		"/core-service=management/access=authorization/constraint=application-classification/type=*",
		"/core-service=management/access=authorization/constraint=sensitivity-classification/type=*"})
@ResourceType("type")
public class Type<T extends Type<T>> extends HashMap {

	private String key;
	private PropertyChangeSupport pcs;
	private TypeResources subresources = new TypeResources();

	public Type(String key) {
		super();
		this.key = key;
	}

	public String getKey() {
		return this.key;
	}

	/**
	 * Adds a property change listener
	 */
	public void addPropertyChangeListener(PropertyChangeListener listener) {
		if (null == this.pcs)
			this.pcs = new PropertyChangeSupport(this);
		this.pcs.addPropertyChangeListener(listener);
	}

	/**
	 * Removes a property change listener
	 */
	public void removePropertyChangeListener(PropertyChangeListener listener) {
		if (this.pcs != null)
			this.pcs.removePropertyChangeListener(listener);
	}

	public TypeResources subresources() {
		return this.subresources;
	}

	/**
	 * Add all Classification objects to this subresource
	 * 
	 * @return this
	 * @param value
	 *            List of Classification objects.
	 */
	@SuppressWarnings("unchecked")
	public T classifications(List<Classification> value) {
		this.subresources.classifications = value;
		return (T) this;
	}

	/**
	 * Add the Classification object to the list of subresources
	 * 
	 * @param value
	 *            The Classification to add
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T classification(Classification value) {
		this.subresources.classifications.add(value);
		return (T) this;
	}

	/**
	 * Create and configure a Classification object to the list of subresources
	 * 
	 * @param key
	 *            The key for the Classification resource
	 * @param config
	 *            The ClassificationConsumer to use
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T classification(String childKey, ClassificationConsumer consumer) {
		Classification<? extends Classification> child = new Classification<>(
				childKey);
		if (consumer != null) {
			consumer.accept(child);
		}
		classification(child);
		return (T) this;
	}

	/**
	 * Create and configure a Classification object to the list of subresources
	 * 
	 * @param key
	 *            The key for the Classification resource
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T classification(String childKey) {
		classification(childKey, null);
		return (T) this;
	}

	/**
	 * Install a supplied Classification object to the list of subresources
	 */
	@SuppressWarnings("unchecked")
	public T classification(ClassificationSupplier supplier) {
		classification(supplier.get());
		return (T) this;
	}

	/**
	 * Child mutators for Type
	 */
	public static class TypeResources {
		/**
		 * Configuration of an application classification constraint.
		 */
		private List<Classification> classifications = new java.util.ArrayList<>();

		/**
		 * Get the list of Classification resources
		 * 
		 * @return the list of resources
		 */
		@Subresource
		public List<Classification> classifications() {
			return this.classifications;
		}

		public Classification classification(String key) {
			return this.classifications.stream()
					.filter(e -> e.getKey().equals(key)).findFirst()
					.orElse(null);
		}
	}
}