package org.wildfly.swarm.config.management.security_realm;

import org.wildfly.swarm.config.runtime.Address;
import java.util.HashMap;
import org.wildfly.swarm.config.runtime.ResourceType;
import org.wildfly.swarm.config.runtime.Implicit;
import java.beans.PropertyChangeSupport;
import java.beans.PropertyChangeListener;
import java.util.List;
import org.wildfly.swarm.config.runtime.Subresource;
import org.wildfly.swarm.config.management.security_realm.server_identity.KeytabConsumer;
import org.wildfly.swarm.config.management.security_realm.server_identity.KeytabSupplier;
import org.wildfly.swarm.config.management.security_realm.server_identity.Keytab;
import org.wildfly.swarm.config.runtime.ModelNodeBinding;
/**
 * Configuration for the Kerberos identity of a server or host controller.
 */
@Address("/core-service=management/security-realm=*/server-identity=kerberos")
@ResourceType("server-identity")
@Implicit
public class KerberosServerIdentity<T extends KerberosServerIdentity<T>>
		extends
			HashMap {

	private String key;
	private PropertyChangeSupport pcs;
	private KerberosServerIdentityResources subresources = new KerberosServerIdentityResources();

	public KerberosServerIdentity() {
		super();
		this.key = "kerberos";
		this.pcs = new PropertyChangeSupport(this);
	}

	public String getKey() {
		return this.key;
	}

	/**
	 * Adds a property change listener
	 */
	public void addPropertyChangeListener(PropertyChangeListener listener) {
		if (null == this.pcs)
			this.pcs = new PropertyChangeSupport(this);
		this.pcs.addPropertyChangeListener(listener);
	}

	/**
	 * Removes a property change listener
	 */
	public void removePropertyChangeListener(PropertyChangeListener listener) {
		if (this.pcs != null)
			this.pcs.removePropertyChangeListener(listener);
	}

	public KerberosServerIdentityResources subresources() {
		return this.subresources;
	}

	/**
	 * Add all Keytab objects to this subresource
	 * 
	 * @return this
	 * @param value
	 *            List of Keytab objects.
	 */
	@SuppressWarnings("unchecked")
	public T keytabs(List<Keytab> value) {
		this.subresources.keytabs = value;
		return (T) this;
	}

	/**
	 * Add the Keytab object to the list of subresources
	 * 
	 * @param value
	 *            The Keytab to add
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T keytab(Keytab value) {
		this.subresources.keytabs.add(value);
		return (T) this;
	}

	/**
	 * Create and configure a Keytab object to the list of subresources
	 * 
	 * @param key
	 *            The key for the Keytab resource
	 * @param config
	 *            The KeytabConsumer to use
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T keytab(String childKey, KeytabConsumer consumer) {
		Keytab<? extends Keytab> child = new Keytab<>(childKey);
		if (consumer != null) {
			consumer.accept(child);
		}
		keytab(child);
		return (T) this;
	}

	/**
	 * Create and configure a Keytab object to the list of subresources
	 * 
	 * @param key
	 *            The key for the Keytab resource
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T keytab(String childKey) {
		keytab(childKey, null);
		return (T) this;
	}

	/**
	 * Install a supplied Keytab object to the list of subresources
	 */
	@SuppressWarnings("unchecked")
	public T keytab(KeytabSupplier supplier) {
		keytab(supplier.get());
		return (T) this;
	}

	/**
	 * Child mutators for KerberosServerIdentity
	 */
	public static class KerberosServerIdentityResources {
		/**
		 * Configuration of a keytab to use to represent a server or host
		 * controllers identity.
		 */
		private List<Keytab> keytabs = new java.util.ArrayList<>();

		/**
		 * Get the list of Keytab resources
		 * 
		 * @return the list of resources
		 */
		@Subresource
		public List<Keytab> keytabs() {
			return this.keytabs;
		}

		public Keytab keytab(String key) {
			return this.keytabs.stream().filter(e -> e.getKey().equals(key))
					.findFirst().orElse(null);
		}
	}
}