package org.wildfly.swarm.config.resource.adapters.resource_adapter;

import org.wildfly.swarm.config.runtime.Address;
import java.util.HashMap;
import org.wildfly.swarm.config.runtime.ResourceType;
import java.beans.PropertyChangeSupport;
import java.beans.PropertyChangeListener;
import java.util.List;
import org.wildfly.swarm.config.runtime.Subresource;
import org.wildfly.swarm.config.resource.adapters.resource_adapter.ConfigPropertiesConsumer;
import org.wildfly.swarm.config.resource.adapters.resource_adapter.ConfigPropertiesSupplier;
import org.wildfly.swarm.config.resource.adapters.resource_adapter.ConfigProperties;
import org.wildfly.swarm.config.runtime.ModelNodeBinding;
/**
 * Specifies an administration object.
 */
@Address("/subsystem=resource-adapters/resource-adapter=*/admin-objects=*")
@ResourceType("admin-objects")
public class AdminObjects<T extends AdminObjects<T>> extends HashMap {

	private String key;
	private PropertyChangeSupport pcs;
	private AdminObjectsResources subresources = new AdminObjectsResources();
	private String className;
	private Boolean enabled;
	private String jndiName;
	private Boolean useJavaContext;

	public AdminObjects(String key) {
		super();
		this.key = key;
	}

	public String getKey() {
		return this.key;
	}

	/**
	 * Adds a property change listener
	 */
	public void addPropertyChangeListener(PropertyChangeListener listener) {
		if (null == this.pcs)
			this.pcs = new PropertyChangeSupport(this);
		this.pcs.addPropertyChangeListener(listener);
	}

	/**
	 * Removes a property change listener
	 */
	public void removePropertyChangeListener(PropertyChangeListener listener) {
		if (this.pcs != null)
			this.pcs.removePropertyChangeListener(listener);
	}

	public AdminObjectsResources subresources() {
		return this.subresources;
	}

	/**
	 * Add all ConfigProperties objects to this subresource
	 * 
	 * @return this
	 * @param value
	 *            List of ConfigProperties objects.
	 */
	@SuppressWarnings("unchecked")
	public T configProperties(List<ConfigProperties> value) {
		this.subresources.configProperties = value;
		return (T) this;
	}

	/**
	 * Add the ConfigProperties object to the list of subresources
	 * 
	 * @param value
	 *            The ConfigProperties to add
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T configProperties(ConfigProperties value) {
		this.subresources.configProperties.add(value);
		return (T) this;
	}

	/**
	 * Create and configure a ConfigProperties object to the list of
	 * subresources
	 * 
	 * @param key
	 *            The key for the ConfigProperties resource
	 * @param config
	 *            The ConfigPropertiesConsumer to use
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T configProperties(String childKey, ConfigPropertiesConsumer consumer) {
		ConfigProperties<? extends ConfigProperties> child = new ConfigProperties<>(
				childKey);
		if (consumer != null) {
			consumer.accept(child);
		}
		configProperties(child);
		return (T) this;
	}

	/**
	 * Create and configure a ConfigProperties object to the list of
	 * subresources
	 * 
	 * @param key
	 *            The key for the ConfigProperties resource
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T configProperties(String childKey) {
		configProperties(childKey, null);
		return (T) this;
	}

	/**
	 * Install a supplied ConfigProperties object to the list of subresources
	 */
	@SuppressWarnings("unchecked")
	public T configProperties(ConfigPropertiesSupplier supplier) {
		configProperties(supplier.get());
		return (T) this;
	}

	/**
	 * Child mutators for AdminObjects
	 */
	public static class AdminObjectsResources {
		/**
		 * A custom defined config property.
		 */
		private List<ConfigProperties> configProperties = new java.util.ArrayList<>();

		/**
		 * Get the list of ConfigProperties resources
		 * 
		 * @return the list of resources
		 */
		@Subresource
		public List<ConfigProperties> configProperties() {
			return this.configProperties;
		}

		public ConfigProperties configProperties(String key) {
			return this.configProperties.stream()
					.filter(e -> e.getKey().equals(key)).findFirst()
					.orElse(null);
		}
	}

	/**
	 * Specifies the fully qualified class name of an administration object.
	 */
	@ModelNodeBinding(detypedName = "class-name")
	public String className() {
		return this.className;
	}

	/**
	 * Specifies the fully qualified class name of an administration object.
	 */
	@SuppressWarnings("unchecked")
	public T className(String value) {
		Object oldValue = this.className;
		this.className = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("className", oldValue, value);
		return (T) this;
	}

	/**
	 * Specifies if the administration object should be enabled.
	 */
	@ModelNodeBinding(detypedName = "enabled")
	public Boolean enabled() {
		return this.enabled;
	}

	/**
	 * Specifies if the administration object should be enabled.
	 */
	@SuppressWarnings("unchecked")
	public T enabled(Boolean value) {
		Object oldValue = this.enabled;
		this.enabled = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("enabled", oldValue, value);
		return (T) this;
	}

	/**
	 * Specifies the JNDI name for the administration object.
	 */
	@ModelNodeBinding(detypedName = "jndi-name")
	public String jndiName() {
		return this.jndiName;
	}

	/**
	 * Specifies the JNDI name for the administration object.
	 */
	@SuppressWarnings("unchecked")
	public T jndiName(String value) {
		Object oldValue = this.jndiName;
		this.jndiName = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("jndiName", oldValue, value);
		return (T) this;
	}

	/**
	 * Setting this to false will bind the object into global JNDI.
	 */
	@ModelNodeBinding(detypedName = "use-java-context")
	public Boolean useJavaContext() {
		return this.useJavaContext;
	}

	/**
	 * Setting this to false will bind the object into global JNDI.
	 */
	@SuppressWarnings("unchecked")
	public T useJavaContext(Boolean value) {
		Object oldValue = this.useJavaContext;
		this.useJavaContext = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("useJavaContext", oldValue, value);
		return (T) this;
	}
}