package org.wildfly.swarm.config;

import org.wildfly.swarm.config.runtime.Address;
import java.util.HashMap;
import org.wildfly.swarm.config.runtime.ResourceType;
import org.wildfly.swarm.config.runtime.Implicit;
import java.beans.PropertyChangeSupport;
import java.beans.PropertyChangeListener;
import java.util.List;
import org.wildfly.swarm.config.runtime.Subresource;
import org.wildfly.swarm.config.naming.BindingConsumer;
import org.wildfly.swarm.config.naming.BindingSupplier;
import org.wildfly.swarm.config.naming.Binding;
import org.wildfly.swarm.config.naming.RemoteNamingService;
import org.wildfly.swarm.config.naming.RemoteNamingServiceConsumer;
import org.wildfly.swarm.config.naming.RemoteNamingServiceSupplier;
import org.wildfly.swarm.config.runtime.ModelNodeBinding;
/**
 * The configuration of the naming subsystem.
 */
@Address("/subsystem=naming")
@ResourceType("subsystem")
@Implicit
public class Naming<T extends Naming<T>> extends HashMap {

	private String key;
	private PropertyChangeSupport pcs;
	private NamingResources subresources = new NamingResources();

	public Naming() {
		super();
		this.key = "naming";
		this.pcs = new PropertyChangeSupport(this);
	}

	public String getKey() {
		return this.key;
	}

	/**
	 * Adds a property change listener
	 */
	public void addPropertyChangeListener(PropertyChangeListener listener) {
		if (null == this.pcs)
			this.pcs = new PropertyChangeSupport(this);
		this.pcs.addPropertyChangeListener(listener);
	}

	/**
	 * Removes a property change listener
	 */
	public void removePropertyChangeListener(PropertyChangeListener listener) {
		if (this.pcs != null)
			this.pcs.removePropertyChangeListener(listener);
	}

	public NamingResources subresources() {
		return this.subresources;
	}

	/**
	 * Add all Binding objects to this subresource
	 * 
	 * @return this
	 * @param value
	 *            List of Binding objects.
	 */
	@SuppressWarnings("unchecked")
	public T bindings(List<Binding> value) {
		this.subresources.bindings = value;
		return (T) this;
	}

	/**
	 * Add the Binding object to the list of subresources
	 * 
	 * @param value
	 *            The Binding to add
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T binding(Binding value) {
		this.subresources.bindings.add(value);
		return (T) this;
	}

	/**
	 * Create and configure a Binding object to the list of subresources
	 * 
	 * @param key
	 *            The key for the Binding resource
	 * @param config
	 *            The BindingConsumer to use
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T binding(String childKey, BindingConsumer consumer) {
		Binding<? extends Binding> child = new Binding<>(childKey);
		if (consumer != null) {
			consumer.accept(child);
		}
		binding(child);
		return (T) this;
	}

	/**
	 * Create and configure a Binding object to the list of subresources
	 * 
	 * @param key
	 *            The key for the Binding resource
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T binding(String childKey) {
		binding(childKey, null);
		return (T) this;
	}

	/**
	 * Install a supplied Binding object to the list of subresources
	 */
	@SuppressWarnings("unchecked")
	public T binding(BindingSupplier supplier) {
		binding(supplier.get());
		return (T) this;
	}

	/**
	 * The remote naming server
	 */
	@SuppressWarnings("unchecked")
	public T remoteNamingService(RemoteNamingService value) {
		this.subresources.remoteNamingService = value;
		return (T) this;
	}

	/**
	 * The remote naming server
	 */
	@SuppressWarnings("unchecked")
	public T remoteNamingService(RemoteNamingServiceConsumer consumer) {
		RemoteNamingService<? extends RemoteNamingService> child = new RemoteNamingService<>();
		if (consumer != null) {
			consumer.accept(child);
		}
		this.subresources.remoteNamingService = child;
		return (T) this;
	}

	/**
	 * The remote naming server
	 */
	@SuppressWarnings("unchecked")
	public T remoteNamingService() {
		RemoteNamingService<? extends RemoteNamingService> child = new RemoteNamingService<>();
		this.subresources.remoteNamingService = child;
		return (T) this;
	}

	/**
	 * The remote naming server
	 */
	@SuppressWarnings("unchecked")
	public T remoteNamingService(RemoteNamingServiceSupplier supplier) {
		this.subresources.remoteNamingService = supplier.get();
		return (T) this;
	}

	/**
	 * Child mutators for Naming
	 */
	public static class NamingResources {
		/**
		 * JNDI bindings for primitive types
		 */
		private List<Binding> bindings = new java.util.ArrayList<>();
		private RemoteNamingService remoteNamingService;

		/**
		 * Get the list of Binding resources
		 * 
		 * @return the list of resources
		 */
		@Subresource
		public List<Binding> bindings() {
			return this.bindings;
		}

		public Binding binding(String key) {
			return this.bindings.stream().filter(e -> e.getKey().equals(key))
					.findFirst().orElse(null);
		}
		/**
		 * The remote naming server
		 */
		@Subresource
		public RemoteNamingService remoteNamingService() {
			return this.remoteNamingService;
		}
	}
}