package org.wildfly.swarm.config.logging;

import org.wildfly.swarm.config.runtime.Addresses;
import java.util.HashMap;
import org.wildfly.swarm.config.runtime.ResourceType;
import java.beans.PropertyChangeSupport;
import java.beans.PropertyChangeListener;
import org.wildfly.swarm.config.runtime.ModelNodeBinding;
import java.util.List;
import java.util.Arrays;
import org.wildfly.swarm.config.logging.Level;
/**
 * Defines a logger category.
 */
@Addresses({"/subsystem=logging/logger=*",
		"/subsystem=logging/logging-profile=*/logger=*"})
@ResourceType("logger")
public class Logger<T extends Logger<T>> extends HashMap {

	private String key;
	private PropertyChangeSupport pcs;
	private String category;
	private String filterSpec;
	private List<String> handlers;
	private Level level;
	private Boolean useParentHandlers;

	public Logger(String key) {
		super();
		this.key = key;
	}

	public String getKey() {
		return this.key;
	}

	/**
	 * Adds a property change listener
	 */
	public void addPropertyChangeListener(PropertyChangeListener listener) {
		if (null == this.pcs)
			this.pcs = new PropertyChangeSupport(this);
		this.pcs.addPropertyChangeListener(listener);
	}

	/**
	 * Removes a property change listener
	 */
	public void removePropertyChangeListener(PropertyChangeListener listener) {
		if (this.pcs != null)
			this.pcs.removePropertyChangeListener(listener);
	}

	/**
	 * Specifies the category for the logger.
	 */
	@ModelNodeBinding(detypedName = "category")
	public String category() {
		return this.category;
	}

	/**
	 * Specifies the category for the logger.
	 */
	@SuppressWarnings("unchecked")
	public T category(String value) {
		Object oldValue = this.category;
		this.category = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("category", oldValue, value);
		return (T) this;
	}

	/**
	 * A filter expression value to define a filter. Example for a filter that
	 * does not match a pattern: not(match("JBAS.*"))
	 */
	@ModelNodeBinding(detypedName = "filter-spec")
	public String filterSpec() {
		return this.filterSpec;
	}

	/**
	 * A filter expression value to define a filter. Example for a filter that
	 * does not match a pattern: not(match("JBAS.*"))
	 */
	@SuppressWarnings("unchecked")
	public T filterSpec(String value) {
		Object oldValue = this.filterSpec;
		this.filterSpec = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("filterSpec", oldValue, value);
		return (T) this;
	}

	/**
	 * The handlers associated with the logger.
	 */
	@ModelNodeBinding(detypedName = "handlers")
	public List<String> handlers() {
		return this.handlers;
	}

	/**
	 * The handlers associated with the logger.
	 */
	@SuppressWarnings("unchecked")
	public T handlers(List<String> value) {
		Object oldValue = this.handlers;
		this.handlers = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("handlers", oldValue, value);
		return (T) this;
	}

	/**
	 * The handlers associated with the logger.
	 */
	@SuppressWarnings("unchecked")
	public T handler(String value) {
		if (this.handlers == null) {
			this.handlers = new java.util.ArrayList<>();
		}
		this.handlers.add(value);
		return (T) this;
	}

	/**
	 * The handlers associated with the logger.
	 */
	@SuppressWarnings("unchecked")
	public T handlers(String... args) {
		handlers(Arrays.asList(args));
		return (T) this;
	}

	/**
	 * The log level specifying which message levels will be logged by the
	 * logger. Message levels lower than this value will be discarded.
	 */
	@ModelNodeBinding(detypedName = "level")
	public Level level() {
		return this.level;
	}

	/**
	 * The log level specifying which message levels will be logged by the
	 * logger. Message levels lower than this value will be discarded.
	 */
	@SuppressWarnings("unchecked")
	public T level(Level value) {
		Object oldValue = this.level;
		this.level = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("level", oldValue, value);
		return (T) this;
	}

	/**
	 * Specifies whether or not this logger should send its output to it's
	 * parent Logger.
	 */
	@ModelNodeBinding(detypedName = "use-parent-handlers")
	public Boolean useParentHandlers() {
		return this.useParentHandlers;
	}

	/**
	 * Specifies whether or not this logger should send its output to it's
	 * parent Logger.
	 */
	@SuppressWarnings("unchecked")
	public T useParentHandlers(Boolean value) {
		Object oldValue = this.useParentHandlers;
		this.useParentHandlers = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("useParentHandlers", oldValue, value);
		return (T) this;
	}
}