package org.wildfly.swarm.config;

import org.wildfly.swarm.config.runtime.Address;
import java.util.HashMap;
import org.wildfly.swarm.config.runtime.ResourceType;
import org.wildfly.swarm.config.runtime.Implicit;
import java.beans.PropertyChangeSupport;
import java.beans.PropertyChangeListener;
import org.wildfly.swarm.config.runtime.ModelNodeBinding;
import java.util.Arrays;
/**
 * The configuration of the JPA subsystem.
 */
@Address("/subsystem=jpa")
@ResourceType("subsystem")
@Implicit
public class JPA<T extends JPA<T>> extends HashMap {

	private String key;
	private PropertyChangeSupport pcs;
	private String defaultDatasource;
	private DefaultExtendedPersistenceInheritance defaultExtendedPersistenceInheritance;

	public JPA() {
		super();
		this.key = "jpa";
		this.pcs = new PropertyChangeSupport(this);
	}

	public String getKey() {
		return this.key;
	}

	/**
	 * Adds a property change listener
	 */
	public void addPropertyChangeListener(PropertyChangeListener listener) {
		if (null == this.pcs)
			this.pcs = new PropertyChangeSupport(this);
		this.pcs.addPropertyChangeListener(listener);
	}

	/**
	 * Removes a property change listener
	 */
	public void removePropertyChangeListener(PropertyChangeListener listener) {
		if (this.pcs != null)
			this.pcs.removePropertyChangeListener(listener);
	}

	public static enum DefaultExtendedPersistenceInheritance {
		DEEP("DEEP"), SHALLOW("SHALLOW");
		private final String allowedValue;

		/**
		 * Returns the allowed value for the management model.
		 * 
		 * @return the allowed model value
		 */
		public String getAllowedValue() {
			return allowedValue;
		}

		DefaultExtendedPersistenceInheritance(String allowedValue) {
			this.allowedValue = allowedValue;
		}

		@Override
		public String toString() {
			return allowedValue;
		}
	}

	/**
	 * The name of the default global datasource.
	 */
	@ModelNodeBinding(detypedName = "default-datasource")
	public String defaultDatasource() {
		return this.defaultDatasource;
	}

	/**
	 * The name of the default global datasource.
	 */
	@SuppressWarnings("unchecked")
	public T defaultDatasource(String value) {
		Object oldValue = this.defaultDatasource;
		this.defaultDatasource = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("defaultDatasource", oldValue, value);
		return (T) this;
	}

	/**
	 * Controls how JPA extended persistence context (XPC) inheritance is
	 * performed. 'DEEP' shares the extended persistence context at top bean
	 * level. 'SHALLOW' the extended persistece context is only shared with the
	 * parent bean (never with sibling beans).
	 */
	@ModelNodeBinding(detypedName = "default-extended-persistence-inheritance")
	public DefaultExtendedPersistenceInheritance defaultExtendedPersistenceInheritance() {
		return this.defaultExtendedPersistenceInheritance;
	}

	/**
	 * Controls how JPA extended persistence context (XPC) inheritance is
	 * performed. 'DEEP' shares the extended persistence context at top bean
	 * level. 'SHALLOW' the extended persistece context is only shared with the
	 * parent bean (never with sibling beans).
	 */
	@SuppressWarnings("unchecked")
	public T defaultExtendedPersistenceInheritance(
			DefaultExtendedPersistenceInheritance value) {
		Object oldValue = this.defaultExtendedPersistenceInheritance;
		this.defaultExtendedPersistenceInheritance = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange(
					"defaultExtendedPersistenceInheritance", oldValue, value);
		return (T) this;
	}
}